(ns mulify.admin.openssl
  (:require [clojure.java.shell :refer [sh]]
            [clojure.string :as str]))

(def work-dir "/tmp")
(def private-key-file (str work-dir "/" "private.key"))
(def server-csr-file (str work-dir "/" "server.csr"))
(def server-crt-file (str work-dir "/" "server.crt"))

(def server-p12-file (str work-dir "/" "server.p12"))


(def ^:dynamic *debug* false)

(defn gen-private-key [& {:keys [length passout]
                          :or {length 2048}}]
  (let [{:keys [exit out err]} (cond-> ["openssl" "genrsa" ]
                                 passout (concat ["-aes256" "-passout" (str "pass:" passout)
                                                  ])
                                 length (concat [(str length)])
                                 *debug* (doto (->> (str/join " ") println))
                                 :exec (->> (apply sh)))]
    (if (= 0 exit)
      (do (println "created key:" private-key-file)
          (spit private-key-file out))
      (println err))))

(def CN "*.lb.anypointdns.net")

(defn gen-certificate-signing-request [& {:keys [key-file passin config subject cn]
                                          :or {key-file private-key-file
                                               cn "localhost"}}]
  (let [subject (str "/C=JP/ST=Tokyo/L=Chiyoda/O=My Corp/OU=IT/CN=" cn)
        {:keys [exit out err]} (cond-> ["openssl" "req" "-new"]
                                 config (concat ["-config" config])
                                 key-file (concat ["-key" key-file])
                                 :exec (concat ["-subj" subject])
                                 passin (concat ["-passin" (str "pass:" passin)])
                                 *debug* (doto (->> (str/join " ") println))
                                 :exec (->> (apply sh)))]
    (if (= 0 exit)
      (do (println "created csr:" server-csr-file)
          (spit server-csr-file out))
      (println err))))

(defn self-certify [& {:keys [key-file csr-file passin days]
                       :or {key-file private-key-file
                            days 365
                            csr-file server-csr-file}}]
  (let [{:keys [exit out err]} (cond-> ["openssl" "x509" "-req"]
                                 key-file (concat ["-signkey" key-file])
                                 days (concat ["-days" (str days)])
                                 csr-file (concat ["-in" csr-file])
                                 :exec (concat ["-passin" (str "pass:" passin)])
                                 *debug* (doto (->> (str/join " ") println))                                 
                                 :exec (->> (apply sh)))]
    (if (= 0 exit)
      (do (println "created crt:" server-crt-file)
          (spit server-crt-file out))
      (println err))))


(defn gen-self-certification [key-pass]
  (gen-private-key :passout key-pass)
  (gen-certificate-signing-request :passin key-pass)
  (self-certify :passin key-pass))

(defn gen-self-certification-without-passout []
  (gen-private-key)
  (gen-certificate-signing-request)
  (self-certify ))



;; X509
(defn x509->pkcs12 [& {:keys [pem-file key-file passin passout p12-out]
                       :or {pem-file server-crt-file
                            key-file private-key-file
                            p12-out server-p12-file}}]
  (let [
        
        {:keys [exit out err]} (cond-> ["openssl" "pkcs12" "-export"]
                                 pem-file (concat ["-in" pem-file])
                                 key-file (concat ["-inkey" key-file])
                                 :exec (concat ["-out" p12-out])
                                 :exec  (concat ["-passin" (str "pass:" passin)])
                                 :exec  (concat ["-passout" (str "pass:" passout)])
                                 *debug* (doto (->> (str/join " ") println))                                                                  
                                 :exec (->> (apply sh)))]
    (if (= 0 exit)
      (println "created pkcs12:" p12-out)
      (println err))))

(defn gen-self-certification-pkcs12 [key-pass & [store-pass]]
  ;; Java does not support the difference of keypass and storepass.
  (gen-self-certification key-pass)
  (x509->pkcs12 :passin key-pass :passout (or store-pass key-pass)))

;; PKCS12
(defn pkcs12->x509 [& {:keys [passin passout p12-file keys? certs? cacerts?]}]
  (let [{:keys [exit out err]} (cond-> ["openssl" "pkcs12" "-nodes"]
                                 p12-file (concat ["-in" p12-file])
                                 (some? keys?) (concat ["-nocerts"])
                                 (some? certs?) (concat ["-nokeys"])
                                 (some? cacerts?) (concat ["-cacerts" "-nokeys" "-nocerts"])
                                 :exec  (concat ["-passin" (str "pass:" passin)])
                                 :exec  (concat ["-passout" (str "pass:" passout)])
                                 :exec (concat ["-info"])
                                 :exec (->> (apply sh)))]
    (if (= 0 exit)
      out
      (println err))))


