(ns mulify.salesforce
  (:require [clojure.data.xml :as dx]
            [silvur.util :refer [uuid]]
            [mulify.core :refer (defprocess >camel)]
            [clojure.spec.alpha :as s]
            [clojure.spec.test.alpha :as t]
            [mulify.dataweave :as dw]))

(defn string->cdata [x]
  (if (string? x) (dx/cdata x) x))

(defprocess sfdc-config {:name "Salesforce_Config"})
(defprocess basic-connection {:username "" :password "" :security-token ""})

(defprocess query {:config-ref "Salesforce_Config"}
  :update-fn (fn [x]
               x
               ))
(defprocess salesforce-query {} :update-fn string->cdata)
(defprocess parameters {} :update-fn string->cdata)
(defprocess replay-channel-listener)

(defprocess config-with-oauth-connection)
(defprocess oauth-authorization-code {:consumer-key "" :consumer-secret ""})
(defprocess auth-callback-config {:listener-config "" :callback-path "" :authorize-path ""})


(defn sfdc-config* [{:keys [name username password security-token] :as args}]
  (sfdc-config {:name name}
    (basic-connection (dissoc args :name))))

(defn query* [sf-config query-body & params]
  ;; Assume sf-config is XML tree
  (query {:config-ref (get-in sf-config [:attrs :name])}
    (salesforce-query (dx/cdata query-body))
    (parameters (dw/fx (into {} (map vec (partition-all 2 params)))))))


;; Configuration
(s/fdef sfdc-config
  :args (s/cat
          :attr (s/keys :req-un [::name])
          :basic-connection #(-> % :tag #{::basic-connection})))

(s/fdef basic-connection
  :args (s/cat :credential (s/keys :req-un [::username ::password ::security-token])))

;; Query
(s/fdef query
  :args (s/cat
          :attr (s/? (s/keys :req-un [::config-ref] :opt-un [::target]))
          :salesforce-query (s/* #(-> % :tag #{::salesforce-query ::parameters}))))

(s/fdef salesforce-query
  :args (s/cat :cdata (s/alt
                        :cdata (partial instance? clojure.data.xml.node.CData)
                        :string string?)))

(s/fdef parameters
  :args (s/alt
          :cdata (partial instance? clojure.data.xml.node.CData)
          :string string?))


(s/fdef config-with-oauth-connection
  :args (s/cat
          :oauth-authorization-code #(-> % :tag #{::oauth-authorization-code})
          :auth-callback-config #(-> % :tag #{::auth-callback-config})))

(s/fdef oauth-authorization-code
  :args (s/cat 
          :attr (s/keys :req-un [::consumer-key ::consumer-secret])))

(s/fdef auth-callback-config
  :args (s/cat 
          :attr (s/keys :req-un [::listener-config ::callback-path ::authorize-path])))


(t/instrument)
