package io.kosong.flink.clojure.functions;

import clojure.java.api.Clojure;
import clojure.lang.APersistentMap;
import clojure.lang.IFn;
import clojure.lang.Keyword;
import clojure.lang.Namespace;
import org.apache.flink.api.common.functions.RichReduceFunction;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.api.java.typeutils.ResultTypeQueryable;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;

public class CljReduceFunction<T> extends RichReduceFunction<T> implements ResultTypeQueryable<T>, CheckpointedFunction {

    private final Namespace namespace;
    private final IFn initFn;
    private final IFn initializeStateFn;
    private final IFn openFn;
    private final IFn closeFn;
    private final IFn reduceFn;
    private final IFn snapshotStateFn;
    private final TypeInformation<T> returnType;

    private transient Object state;
    private transient boolean initialized;

    public CljReduceFunction(APersistentMap args) {
        namespace = (Namespace) Keyword.intern("ns").invoke(args);
        returnType = (TypeInformation) Keyword.intern("returns").invoke(args);
        initFn = (IFn) Keyword.intern("init").invoke(args);
        openFn = (IFn) Keyword.intern("open").invoke(args);
        closeFn = (IFn) Keyword.intern("close").invoke(args);
        initializeStateFn = (IFn) Keyword.intern("initializeState").invoke(args);
        snapshotStateFn = (IFn) Keyword.intern("snapshotState").invoke(args);
        reduceFn = (IFn) Keyword.intern("reduce").invoke(args);
    }

    private void init() {
        Clojure.var("clojure.core/require").invoke(namespace.getName());
        if (initFn != null) {
            state = initFn.invoke(this);
        }
        initialized = true;
    }

    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        if (!initialized) {
            init();
        }
        if (openFn != null) {
            openFn.invoke(this, parameters);
        }
    }

    @Override
    public void close() throws Exception {
        if (closeFn != null) {
            closeFn.invoke(this);
        }
        super.close();
    }

    public TypeInformation getProducedType() {
        return returnType;
    }


    public void snapshotState(FunctionSnapshotContext context) throws Exception {
        if (snapshotStateFn != null) {
            snapshotStateFn.invoke(this, context);
        }
    }

    public void initializeState(FunctionInitializationContext context) throws Exception {
        if (!initialized) {
            init();
        }
        if (initializeStateFn != null) {
            initializeStateFn.invoke(this, context);
        }
    }

    @Override
    public T reduce(T value1, T value2) throws Exception {
        return (T) reduceFn.invoke(this, value1, value2);
    }
}