(ns sputter.state.inmem.node
  (:require [sputter.state.trie.util :refer [split-prefix]]))

(defprotocol Node
  (value [node k]
    "Return the value of this node provided the supplied key remainder
     matches.")
  (attach [node k v]
    "Attach the `v` to this node, at the given key.")
  (reattach [node k child]
    "Replace the child node at `k` with `child`.")
  (descend [node candidate-k]
    "Locate the immediate child pointed to by the given key.
     Return vector of `[child remaining-k consumed-k]` or `nil`."))

(defn node? [x]
  (or (vector? x) (extends? Node (class x))))

(def ^:private empty-branch (into [] (take 17 (repeat nil))))

(declare ->LeafNode ->ExtensionNode)

(extend-type clojure.lang.IPersistentVector
  Node
  (descend [this [idx & rest-k]]
    (when idx
      [(this (int idx)) rest-k (list idx)]))
  (value    [this k]           (when (empty? k) (peek this)))
  (reattach [this [idx] child]
    (println "OMG TYPE" (type idx))
    (assoc this (int idx) child))
  (attach   [this [idx & rest-k] v]
    (if idx
      (assoc this (int idx) (->LeafNode rest-k v))
      (assoc this (dec (count this)) v))))

(defrecord LeafNode [key value]
  Node
  (descend [_ _]           nil)
  (value   [_ candidate-k] (when (= candidate-k key)
                             value))
  (attach  [this input-k v]
    (if (= input-k key)
      (assoc this :value v)
      (let [[pre key-suff input-suff] (split-prefix key input-k)]
        (-> empty-branch
            (attach input-suff v)
            (attach key-suff   value)
            (cond->> pre (->ExtensionNode pre)))))))

(defrecord ExtensionNode [key child]
  Node
  (descend [this candidate-k]
    (let [[prefix tail] (split-at (count key) candidate-k)]
      (when (= prefix key)
        [child tail prefix])))
  (value    [_ k]          nil)
  (reattach [this _ child] (assoc this :child child))
  (attach   [this input-k v]
    (let [[pre key-suff input-suff] (split-prefix key input-k)
          [nibble & rest-k]         key-suff
          me                        (if rest-k
                                      (assoc this :key rest-k)
                                      child)]
        (-> empty-branch
            (attach    input-suff    v)
            (reattach  (list nibble) me)
            (cond->> pre (->ExtensionNode pre))))))

(extend-type nil
  Node
  (descend [_ _] nil)
  (value   [_ _] nil)
  (attach  [_ k v]
    (->LeafNode k v)))

(defprotocol -NavigableNode
  (-children [node]))

(extend-protocol -NavigableNode
  clojure.lang.IPersistentVector
  (-children [node] (range (dec (count node))))
  LeafNode
  (-children [_] nil)
  ExtensionNode
  (-children [_] [:child]))

(defn collect [node state outer inner parent]
  (let [[state node]
        (reduce
         (fn [[state node] child-k]
           (let [[state child] (inner state (get node child-k) node)]
             [state (assoc node child-k child)]))
         [state node]
         (-children node))]
    (outer state node parent)))
