(ns slipway.jetty9.websockets
  "Jetty9 impl of the Websockets API + handler.

  Dervied from:
    * https://github.com/sunng87/ring-jetty9-adapter/blob/master/src/ring/adapter/jetty9/websocket.clj"
  (:require [slipway.util :as util]
            [slipway.websockets :as ws])
  (:import (org.eclipse.jetty.server Request Response)
           (org.eclipse.jetty.websocket.api WebSocketAdapter Session RemoteEndpoint WriteCallback)
           (org.eclipse.jetty.websocket.api.extensions ExtensionConfig)
           (org.eclipse.jetty.websocket.server WebSocketHandler)
           (org.eclipse.jetty.websocket.servlet WebSocketServletFactory WebSocketCreator ServletUpgradeRequest)
           (clojure.lang IFn)
           (java.nio ByteBuffer)
           (java.util Locale)))

(def ^:private no-op (constantly nil))

(defn- write-callback
  [{:keys [write-failed write-success]
    :or   {write-failed  no-op
           write-success no-op}}]
  (reify WriteCallback
    (writeFailed [_ throwable]
      (write-failed throwable))
    (writeSuccess [_]
      (write-success))))

(extend-protocol ws/WebSocketSend
  (Class/forName "[B")
  (-send!
    ([ba ws]
     (ws/-send! (ByteBuffer/wrap ba) ws))
    ([ba ws callback]
     (ws/-send! (ByteBuffer/wrap ba) ws callback))))

(extend-protocol ws/WebSocketSend
  ByteBuffer
  (-send!
    ([bb ws]
     (-> ^WebSocketAdapter ws .getRemote (.sendBytes ^ByteBuffer bb)))
    ([bb ws callback]
     (-> ^WebSocketAdapter ws .getRemote (.sendBytes ^ByteBuffer bb ^WriteCallback (write-callback callback)))))

  String
  (-send!
    ([s ws]
     (-> ^WebSocketAdapter ws .getRemote (.sendString ^String s)))
    ([s ws callback]
     (-> ^WebSocketAdapter ws .getRemote (.sendString ^String s ^WriteCallback (write-callback callback)))))

  IFn
  (-send! [f ws]
    (-> ^WebSocketAdapter ws .getRemote f))

  Object
  (send!
    ([this ws]
     (-> ^WebSocketAdapter ws .getRemote
         (.sendString ^RemoteEndpoint (str this))))
    ([this ws callback]
     (-> ^WebSocketAdapter ws .getRemote
         (.sendString ^RemoteEndpoint (str this) ^WriteCallback (write-callback callback))))))

(extend-protocol ws/WebSocketPing
  (Class/forName "[B")
  (-ping! [ba ws] (ws/-ping! (ByteBuffer/wrap ba) ws)))

(extend-protocol ws/WebSocketPing
  ByteBuffer
  (-ping! [bb ws] (-> ^WebSocketAdapter ws .getRemote (.sendPing ^ByteBuffer bb)))

  String
  (-ping! [s ws] (ws/-ping! (.getBytes ^String s "UTF-8") ws))

  Object
  (-ping! [o ws] (ws/-ping! (.getBytes (str o) "UTF-8") ws)))

(extend-protocol util/RequestMapDecoder
  ServletUpgradeRequest
  (build-request-map [request]
    (let [servlet-request  (.getHttpServletRequest request)
          base-request-map {:server-port     (.getServerPort servlet-request)
                            :server-name     (.getServerName servlet-request)
                            :remote-addr     (.getRemoteAddr servlet-request)
                            :uri             (.getRequestURI servlet-request)
                            :query-string    (.getQueryString servlet-request)
                            :scheme          (keyword (.getScheme servlet-request))
                            :request-method  (keyword (.toLowerCase (.getMethod servlet-request) Locale/ENGLISH))
                            :protocol        (.getProtocol servlet-request)
                            :headers         (util/get-headers servlet-request)
                            :ssl-client-cert (first (.getAttribute servlet-request "javax.servlet.request.X509Certificate"))}]
      (assoc base-request-map
             :websocket-subprotocols (into [] (.getSubProtocols request))
             :websocket-extensions (into [] (.getExtensions request))))))

(extend-protocol ws/WebSockets
  WebSocketAdapter
  (send!
    ([this msg]
     (ws/-send! msg this))
    ([this msg callback]
     (ws/-send! msg this callback)))
  (ping!
    ([this]
     (ws/-ping! (ByteBuffer/allocate 0) this))
    ([this msg]
     (ws/-ping! msg this)))
  (close!
    ([this]
     (.close (.getSession this)))
    ([this status-code reason]
     (.close (.getSession this) status-code reason)))
  (remote-addr [this]
    (.getRemoteAddress (.getSession this)))
  (idle-timeout! [this ms]
    (.setIdleTimeout (.getSession this) ms))
  (connected? [this]
    (. this (isConnected)))
  (req-of [this]
    (util/build-request-map (.getUpgradeResponse (.getSession this)))))

(defn proxy-ws-adapter
  [{:keys [on-connect on-error on-text on-close on-bytes]
    :or   {on-connect no-op
           on-error   no-op
           on-text    no-op
           on-close   no-op
           on-bytes   no-op}}]
  (proxy [WebSocketAdapter] []
    (onWebSocketConnect [^Session session]
      (let [^WebSocketAdapter this this]
        (proxy-super onWebSocketConnect session))
      (on-connect this))
    (onWebSocketError [^Throwable e]
      (on-error this e))
    (onWebSocketText [^String message]
      (on-text this message))
    (onWebSocketClose [statusCode ^String reason]
      (let [^WebSocketAdapter this this]
        (proxy-super onWebSocketClose statusCode reason))
      (on-close this statusCode reason))
    (onWebSocketBinary [^bytes payload offset len]
      (on-bytes this payload offset len))))

(defn reify-ws-creator
  [handler]
  (reify WebSocketCreator
    (createWebSocket [_ req resp]
      (let [req-map (util/build-request-map req)]
        (when (ws/upgrade-request? req-map)
          (let [resp-map (handler req-map)]
            (when (ws/upgrade-response? resp-map)
              (let [ws-results (:ws resp-map)]
                (if-let [{:keys [code message headers]} (:error ws-results)]
                  (do (util/set-headers resp headers)
                      (.sendError resp code message))
                  (do
                    (when-let [sp (:subprotocol ws-results)]
                      (.setAcceptedSubProtocol resp sp))
                    (when-let [exts (not-empty (:extensions ws-results))]
                      (.setExtensions resp (mapv #(ExtensionConfig. ^String %) exts)))
                    (proxy-ws-adapter ws-results)))))))))))

(defn proxy-ws-handler
  [handler
   {:keys [ws-max-idle-time
           ws-max-text-message-size]
    :or   {ws-max-idle-time         500000
           ws-max-text-message-size 65536}}]
  (proxy [WebSocketHandler] []
    (configure [^WebSocketServletFactory factory]
      (doto (.getPolicy factory)
        (.setIdleTimeout ws-max-idle-time)
        (.setMaxTextMessageSize ws-max-text-message-size))
      (.setCreator factory (reify-ws-creator handler)))
    (handle [^String target, ^Request request req ^Response res]
      (let [^WebSocketHandler this       this
            ^WebSocketServletFactory wsf (proxy-super getWebSocketFactory)]
        (if (.isUpgradeRequest wsf req res)
          (if (.acceptWebSocket wsf req res)
            (.setHandled request true)
            (when (.isCommitted res)
              (.setHandled request true)))
          (proxy-super handle target request req res))))))
