(ns

    ^{:doc    "Singleton wrappers and helpers for phone-number."
      :author "Paweł Wilk"
      :added  "8.12.4-0"}

    phone-number.util

  (:refer-clojure :exclude [short])

  (:import [com.google.i18n.phonenumbers
            PhoneNumberUtil
            ShortNumberInfo
            geocoding.PhoneNumberOfflineGeocoder
            PhoneNumberToCarrierMapper
            PhoneNumberToTimeZonesMapper
            NumberParseException]))

;; Singletons

(defn instance          {:tag PhoneNumberUtil,              :added  "8.12.4-0"} [] (PhoneNumberUtil/getInstance))
(defn short             {:tag ShortNumberInfo,              :added  "8.12.4-0"} [] (ShortNumberInfo/getInstance))
(defn geo-coder         {:tag PhoneNumberOfflineGeocoder,   :added  "8.12.4-0"} [] (PhoneNumberOfflineGeocoder/getInstance))
(defn carrier-mapper    {:tag PhoneNumberToCarrierMapper,   :added  "8.12.4-0"} [] (PhoneNumberToCarrierMapper/getInstance))
(defn time-zones-mapper {:tag PhoneNumberToTimeZonesMapper, :added  "8.12.4-0"} [] (PhoneNumberToTimeZonesMapper/getInstance))

;; Helpers

(defmacro try-parse
  "Evaluates body and if NumberParseException or NumberFormatException exception is
  caught it returns nil."
  {:added "8.12.4-0"}
  [& body]
  `(try ~@body
        (catch AssertionError        e# nil)
        (catch NumberParseException  e# nil)
        (catch NumberFormatException e# nil)))

(defmacro try-parse-or-false
  "Evaluates body and if NumberParseException or NumberFormatException exception is
  caught it returns false."
  {:added "8.12.4-0"}
  [& body]
  `(try (or (do ~@body) false)
        (catch AssertionError        e# false)
        (catch NumberParseException  e# false)
        (catch NumberFormatException e# false)))

(defmacro try-null
  "Evaluates body and if NullPointerException exception is caught it returns
  nil. Otherwise it returns the value of last expression in the body."
  {:added "8.12.4-0"}
  [& body]
  `(try ~@body
        (catch NullPointerException  e# nil)))

(defn ns-infer
  "Takes a string of namespace name and a keyword. If the given keyword is not
  namespace-qualified it returns a new keyword with the given namespace added. If the
  given keyword is already equipped with a namespace it returns it."
  {:added "8.12.4-0" :tag clojure.lang.Keyword}
  ([^String ns-name
    ^clojure.lang.Keyword k]
   (if (simple-keyword? k)
     (keyword ns-name (name k))
     k))
  ([^String ns-name
    ^clojure.lang.Keyword k
    ^Boolean use-infer]
   (if use-infer (ns-infer ns-name k) k)))

(defn inferred-contains?
  "Just like the contains? but if the keyword is namespace-qualified it also checks if
  the collection contains the same keyword as its key but without a namespace."
  {:added "8.12.4-0" :tag Boolean}
  [^clojure.lang.IPersistentMap coll
   ^clojure.lang.Keyword k]
  (or (contains? coll k)
      (if (simple-keyword? k)
        false
        (contains? coll (keyword (name k))))))

(defn inferred-get
  "Just like the get function but if the keyword is namespace-qualified it first
  attempts to look for the value associated with it. If that fails it uses the
  variant of the keyword without any namespace."
  {:added "8.12.4-0"}
  ([^clojure.lang.IPersistentMap coll
    ^clojure.lang.Keyword k]
   (inferred-get coll k nil))
  ([^clojure.lang.IPersistentMap coll
    ^clojure.lang.Keyword k
    default]
   (if (simple-keyword? k)
     (k coll default)
     ((if (contains? coll k) k (keyword (name k))) coll default))))

(defn fmap-k
  "For each key and value of the given map m calls a function passed as the second
  argument (passing successive keys during calls to it) and generates a map with
  values updated by the results returned by the function."
  {:added "8.12.4-0" :tag clojure.lang.IPersistentMap}
  [^clojure.lang.IFn f
   ^clojure.lang.IPersistentMap m]
  (into (empty m) (for [[k v] m] [k (f k)])))

(defn fmap-k
  "For each key and value of the given map m calls a function passed as the second
  argument (passing successive keys during calls to it) and generates a map with
  values updated by the results returned by the function."
  {:added "8.12.4-0" :tag clojure.lang.IPersistentMap}
  [^clojure.lang.IFn f
   ^clojure.lang.IPersistentMap m]
  (reduce-kv
   (fn [^clojure.lang.IPersistentMap mp k v]
     (assoc mp k (f k)))
   m m))

(defn remove-empty-vals
  "Removes empty values from a map."
  {:added "8.12.4-0" :tag clojure.lang.IPersistentMap}
  [^clojure.lang.IPersistentMap m]
  (reduce-kv
   (fn [^clojure.lang.IPersistentMap mp k v]
     (if (nil? v) (dissoc mp k) mp))
   m m))

(defn- gen-is-sexp
  "For the given keyword k and function name f it generates predicate function
  definition code that compares the result of calling the function on a phone number
  with the keyword. A helper that is used in macros."
  {:added "8.12.4-0"}
  [^clojure.lang.Keyword k
   ^clojure.lang.Symbol  f]
  (let [fn-name (symbol (str "is-" (name k) "?"))]
    (list 'defn fn-name
          {:added (:added (meta (var gen-is-sexp))), :tag 'Boolean
           :doc (str "Returns true when " (name f) " is " k ", false otherwise.\n")}
          (list '[phone-number]
                (list fn-name 'phone-number nil))
          (list '[phone-number region-specification]
                (list 'util/try-parse-or-false
                      (list '= k (list f 'phone-number 'region-specification)))))))

(defmacro gen-is
  "For the given keyword k and function name f uses gen-is-sexp to generate predicate
  function definitions."
  {:added "8.12.4-0"}
  [k f]
  (gen-is-sexp k f))

(defmacro gen-ises
  "Takes a collection of keywords (evaluated) and a function expressed as a symbol (not
  evaluated) and generates bunch of function definitions using gen-is-sexp."
  {:added "8.12.4-0"}
  [coll f]
  (cons 'do (map #(gen-is-sexp % f) (eval coll))))

(defn get-rand-int
  "Like rand-int but optionally uses random number generator."
  {:added "8.12.4-0" :tag 'int}
  ([^long n]
   (rand-int n))
  ([^long n
    ^java.util.Random rng]
   (if (nil? rng)
     (get-rand-int n)
     (if (zero? n) (int n) (.nextInt rng n)))))

(defn random-digits-len
  "For 0 or 1 it returns its argument. For other positive numbers it returns a random
  natural number from 1 to this number (inclusive) in 50% cases. In other 50% cases
  it returns its argument."
  {:added "8.12.4-0" :tag 'long}
  ([^long x
    ^long iteration
    ^Boolean shrink-now]
   (if (zero? x) x
       (if-not shrink-now x
               (if (zero? iteration) 1
                   (if (or (< iteration 6) (zero? (rand-int 2)))
                     (unchecked-inc (rand-int x)) x)))))
  ([^long x
    ^long iteration
    ^Boolean shrink-now
    ^java.util.Random rng]
   (if (nil? rng)
     (random-digits-len x iteration)
     (if (zero? x) x
         (if-not shrink-now x
                 (if (zero? iteration) 1
                     (if (or (< iteration 6) (zero? (get-rand-int 2 rng)))
                       (unchecked-inc (get-rand-int x rng)) x)))))))

(defn gen-digits
  "Generates the given number of random digits and converts all into a single string.
  When the second argument is present it should be an instance of random number
  generator used to get the digits."
  {:added "8.12.4-0" :tag String}
  ([^long num]
   (apply str (repeatedly num #(rand-int 10))))
  ([^long num
    ^java.util.Random rng]
   (if (nil? rng)
     (gen-digits num)
     (apply str (repeatedly num #(.nextInt rng 10))))))

(defn get-rand-nth
  "Returns a random element of the given vector. When the second argument is present it
  should be an instance of random number generator used to get the random position."
  {:added "8.12.4-0" :tag clojure.lang.Keyword}
  ([^clojure.lang.IPersistentVector v]
   (rand-nth v))
  ([^clojure.lang.IPersistentVector v
    ^java.util.Random rng]
   (if (nil? rng)
     (get-rand-nth v)
     (nth v (.nextInt rng (count v))))))

(defn lazy-iterator-seq
  "Returns a lazy sequence as an interface to the given iterable Java object."
  {:added "8.12.4-0" :tag clojure.lang.LazySeq}
  ([^Iterable coll]
   (lazy-iterator-seq coll (.iterator coll)))
  ([^Iterable coll ^java.util.Iterator iter]
   (lazy-seq
    (when (.hasNext iter)
      (cons (.next iter) (lazy-iterator-seq coll iter))))))
