/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.primer.auth;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.base.Strings;
import io.dropwizard.primer.PrimerBundle;
import io.dropwizard.primer.auth.AuthType;
import io.dropwizard.primer.auth.orchestration.KeyOrchestrator;
import io.dropwizard.primer.core.ServiceUser;
import io.dropwizard.primer.core.VerifyResponse;
import io.dropwizard.primer.core.VerifyStaticResponse;
import io.dropwizard.primer.exception.PrimerException;
import io.dropwizard.primer.model.PrimerAuthorization;
import io.dropwizard.primer.model.PrimerAuthorizationMatrix;
import io.dropwizard.primer.model.PrimerBundleConfiguration;
import java.security.Key;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PrimerAuthorizationRegistry {
    private static final Logger log = LoggerFactory.getLogger(PrimerAuthorizationRegistry.class);
    private static Map<String, PrimerAuthorization> authList;
    private static List<String> whiteList;
    private static List<String> urlPatterns;
    private static LoadingCache<String, Optional<Boolean>> blacklistCache;
    private static LoadingCache<TokenKey, JwtClaims> lruCache;
    private static int clockSkew;
    private static String hmacPrivateKey;
    private static KeyOrchestrator keyOrchestrator;

    public static void init(PrimerAuthorizationMatrix matrix, Set<String> whiteListUrls, PrimerBundleConfiguration configuration, KeyOrchestrator jwkKeyOrchestrator) {
        Pattern tokenMatch = Pattern.compile("\\{(([^/])+\\})");
        HashMap<String, PrimerAuthorization> authList = new HashMap<String, PrimerAuthorization>();
        ArrayList<String> urlPatterns = new ArrayList<String>();
        if (matrix != null) {
            if (matrix.getAuthorizations() != null) {
                matrix.getAuthorizations().forEach(auth -> {
                    String pattern = PrimerAuthorizationRegistry.generatePathExpression(auth.getUrl());
                    urlPatterns.add(pattern);
                    authList.put(pattern, (PrimerAuthorization)auth);
                });
            }
            if (matrix.getStaticAuthorizations() != null) {
                matrix.getStaticAuthorizations().forEach(auth -> {
                    String pattern = PrimerAuthorizationRegistry.generatePathExpression(auth.getUrl());
                    urlPatterns.add(pattern);
                    authList.put(pattern, (PrimerAuthorization)auth);
                });
            }
            if (matrix.getAutoAuthorizations() != null) {
                matrix.getAutoAuthorizations().forEach(auth -> {
                    String pattern = PrimerAuthorizationRegistry.generatePathExpression(auth.getUrl());
                    urlPatterns.add(pattern);
                    authList.put(pattern, (PrimerAuthorization)auth);
                });
            }
            urlPatterns.sort((o1, o2) -> tokenMatch.matcher((CharSequence)o2).groupCount() - tokenMatch.matcher((CharSequence)o1).groupCount());
            urlPatterns.sort(Comparator.reverseOrder());
        }
        PrimerAuthorizationRegistry.authList = authList;
        whiteList = PrimerAuthorizationRegistry.primerWhitelistedUrls(whiteListUrls, tokenMatch);
        PrimerAuthorizationRegistry.urlPatterns = urlPatterns;
        blacklistCache = Caffeine.newBuilder().expireAfterWrite((long)configuration.getCacheExpiry(), TimeUnit.SECONDS).maximumSize((long)configuration.getCacheMaxSize()).build(key -> Optional.of(false));
        lruCache = Caffeine.newBuilder().expireAfterWrite((long)configuration.getCacheExpiry(), TimeUnit.SECONDS).maximumSize((long)configuration.getCacheMaxSize()).build(PrimerAuthorizationRegistry::verifyToken);
        clockSkew = configuration.getClockSkew();
        hmacPrivateKey = configuration.getPrivateKey();
        keyOrchestrator = jwkKeyOrchestrator;
    }

    private static JwtConsumer getVerificationJwtConsumer(Key key, int secondsOfAllowedClockSkew) {
        return new JwtConsumerBuilder().setRequireExpirationTime().setAllowedClockSkewInSeconds(secondsOfAllowedClockSkew).setRequireSubject().setSkipDefaultAudienceValidation().setVerificationKey(key).setJwsAlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{"HS512", "RS256"}).build();
    }

    private static List<String> primerWhitelistedUrls(Set<String> whiteListUrls, Pattern tokenMatch) {
        ArrayList<String> whiteList = new ArrayList<String>();
        whiteListUrls.forEach(url -> whiteList.add(PrimerAuthorizationRegistry.generatePathExpression(url)));
        whiteList.sort((o1, o2) -> tokenMatch.matcher((CharSequence)o2).groupCount() - tokenMatch.matcher((CharSequence)o1).groupCount());
        whiteList.sort(Comparator.reverseOrder());
        return whiteList;
    }

    private static String generatePathExpression(String path) {
        return path.replaceAll("\\{(([^/])+\\})", "(([^/])+)");
    }

    public static JwtClaims authorize(String path, String method, String token, AuthType authType, String primerKeyId) {
        return (JwtClaims)lruCache.get((Object)TokenKey.builder().method(method).path(path).token(token).authType(authType).primerKeyId(primerKeyId).build());
    }

    public static boolean isWhilisted(String path) {
        return whiteList.stream().anyMatch(path::matches);
    }

    private static boolean isAuthorized(String id, String method, String role) {
        return authList.get(id).getRoles().contains(role) && authList.get(id).getMethods().contains(method);
    }

    private static JwtClaims verify(JwtClaims jwtClaims, String token, String type) throws PrimerException, MalformedClaimException {
        switch (type) {
            case "dynamic": {
                return PrimerAuthorizationRegistry.verifyDynamic(jwtClaims, token);
            }
            case "static": {
                return PrimerAuthorizationRegistry.verifyStatic(jwtClaims, token);
            }
        }
        log.debug("invalid_token_type type:{} token:{}", (Object)type, (Object)token);
        throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
    }

    private static JwtClaims verifyDynamic(JwtClaims jwtClaims, String token) throws PrimerException, MalformedClaimException {
        boolean result;
        VerifyResponse verifyResponse = PrimerBundle.getPrimerClient().verify(jwtClaims.getIssuer(), jwtClaims.getSubject(), token, ServiceUser.builder().id(jwtClaims.getClaimValueAsString("user_id")).name(jwtClaims.getClaimValueAsString("name")).role(jwtClaims.getClaimValueAsString("role")).build());
        boolean bl = result = !Strings.isNullOrEmpty((String)verifyResponse.getToken()) && !Strings.isNullOrEmpty((String)verifyResponse.getUserId());
        if (!result) {
            log.debug("dynamic_token_validation_failed token:{} verify_response:{}", (Object)token, (Object)verifyResponse);
            PrimerAuthorizationRegistry.blacklist(token);
            throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
        }
        return jwtClaims;
    }

    private static JwtClaims verifyStatic(JwtClaims jwtClaims, String token) throws PrimerException, MalformedClaimException {
        boolean result;
        VerifyStaticResponse verifyStaticResponse = PrimerBundle.getPrimerClient().verify(jwtClaims.getIssuer(), jwtClaims.getSubject(), token, jwtClaims.getClaimValueAsString("role"));
        boolean bl = result = !Strings.isNullOrEmpty((String)verifyStaticResponse.getToken()) && !Strings.isNullOrEmpty((String)verifyStaticResponse.getId());
        if (!result) {
            log.debug("dynamic_token_validation_failed token:{} verify_response:{}", (Object)token, (Object)verifyStaticResponse);
            PrimerAuthorizationRegistry.blacklist(token);
            throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
        }
        return jwtClaims;
    }

    private static JwtClaims verifyConfigAuthToken(TokenKey tokenKey, JwtClaims jwtClaims) throws PrimerException, MalformedClaimException {
        String role = jwtClaims.getClaimValueAsString("role");
        Optional<String> index = urlPatterns.stream().filter(tokenKey.getPath()::matches).findFirst();
        if (!index.isPresent()) {
            log.debug("No index found for {}", (Object)tokenKey);
            throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
        }
        if (!PrimerAuthorizationRegistry.isAuthorized(index.get(), tokenKey.getMethod(), role)) {
            log.debug("Role, method combo check failed for Method={} Role={} Index={}", new Object[]{index.get(), role, tokenKey.getMethod()});
            throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
        }
        switch (authList.get(index.get()).getType()) {
            case "dynamic": {
                return PrimerAuthorizationRegistry.verify(jwtClaims, tokenKey.getToken(), "dynamic");
            }
            case "static": {
                return PrimerAuthorizationRegistry.verify(jwtClaims, tokenKey.getToken(), "static");
            }
            case "auto": {
                String type = jwtClaims.getClaimValueAsString("type");
                return PrimerAuthorizationRegistry.verify(jwtClaims, tokenKey.getToken(), type);
            }
        }
        log.debug("invalid_token_type for index:{} token:{}", (Object)authList.get(index.get()).getType(), (Object)tokenKey);
        throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
    }

    private static JwtClaims verifyAnnotationAuthToken(TokenKey tokenKey, JwtClaims jwtClaims) throws PrimerException, MalformedClaimException {
        String type = jwtClaims.getClaimValueAsString("type");
        return PrimerAuthorizationRegistry.verify(jwtClaims, tokenKey.getToken(), type);
    }

    private static JwtClaims verifyToken(TokenKey tokenKey) throws PrimerException, InvalidJwtException, MalformedClaimException {
        Key key = PrimerAuthorizationRegistry.getPublicKey(tokenKey.getPrimerKeyId());
        JwtConsumer verificationJwtConsumer = PrimerAuthorizationRegistry.getVerificationJwtConsumer(key, clockSkew);
        JwtClaims jwtClaims = verificationJwtConsumer.processToClaims(tokenKey.getToken());
        switch (tokenKey.getAuthType()) {
            case CONFIG: {
                return PrimerAuthorizationRegistry.verifyConfigAuthToken(tokenKey, jwtClaims);
            }
            case ANNOTATION: {
                return PrimerAuthorizationRegistry.verifyAnnotationAuthToken(tokenKey, jwtClaims);
            }
        }
        throw PrimerException.builder().errorCode("PR004").message("Unauthorized").status(401).build();
    }

    private static Key getPublicKey(String rsaKeyId) {
        if (Strings.isNullOrEmpty((String)rsaKeyId)) {
            return keyOrchestrator.getHmacPublicKey(hmacPrivateKey);
        }
        return keyOrchestrator.getRsaPublicKey(rsaKeyId);
    }

    static void blacklist(String token) {
        blacklistCache.put((Object)token, Optional.of(true));
    }

    private PrimerAuthorizationRegistry() {
    }

    private static class TokenKey {
        private String token;
        private String path;
        private String method;
        private AuthType authType;
        private String primerKeyId;

        public static TokenKeyBuilder builder() {
            return new TokenKeyBuilder();
        }

        public String getToken() {
            return this.token;
        }

        public String getPath() {
            return this.path;
        }

        public String getMethod() {
            return this.method;
        }

        public AuthType getAuthType() {
            return this.authType;
        }

        public String getPrimerKeyId() {
            return this.primerKeyId;
        }

        public void setToken(String token) {
            this.token = token;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public void setMethod(String method) {
            this.method = method;
        }

        public void setAuthType(AuthType authType) {
            this.authType = authType;
        }

        public void setPrimerKeyId(String primerKeyId) {
            this.primerKeyId = primerKeyId;
        }

        public TokenKey(String token, String path, String method, AuthType authType, String primerKeyId) {
            this.token = token;
            this.path = path;
            this.method = method;
            this.authType = authType;
            this.primerKeyId = primerKeyId;
        }

        public TokenKey() {
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof TokenKey)) {
                return false;
            }
            TokenKey other = (TokenKey)o;
            if (!other.canEqual(this)) {
                return false;
            }
            String this$token = this.getToken();
            String other$token = other.getToken();
            if (this$token == null ? other$token != null : !this$token.equals(other$token)) {
                return false;
            }
            String this$primerKeyId = this.getPrimerKeyId();
            String other$primerKeyId = other.getPrimerKeyId();
            return !(this$primerKeyId == null ? other$primerKeyId != null : !this$primerKeyId.equals(other$primerKeyId));
        }

        protected boolean canEqual(Object other) {
            return other instanceof TokenKey;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            String $token = this.getToken();
            result = result * 59 + ($token == null ? 43 : $token.hashCode());
            String $primerKeyId = this.getPrimerKeyId();
            result = result * 59 + ($primerKeyId == null ? 43 : $primerKeyId.hashCode());
            return result;
        }

        public String toString() {
            return "PrimerAuthorizationRegistry.TokenKey(token=" + this.getToken() + ", path=" + this.getPath() + ", method=" + this.getMethod() + ", authType=" + (Object)((Object)this.getAuthType()) + ", primerKeyId=" + this.getPrimerKeyId() + ")";
        }

        public static class TokenKeyBuilder {
            private String token;
            private String path;
            private String method;
            private AuthType authType;
            private String primerKeyId;

            TokenKeyBuilder() {
            }

            public TokenKeyBuilder token(String token) {
                this.token = token;
                return this;
            }

            public TokenKeyBuilder path(String path) {
                this.path = path;
                return this;
            }

            public TokenKeyBuilder method(String method) {
                this.method = method;
                return this;
            }

            public TokenKeyBuilder authType(AuthType authType) {
                this.authType = authType;
                return this;
            }

            public TokenKeyBuilder primerKeyId(String primerKeyId) {
                this.primerKeyId = primerKeyId;
                return this;
            }

            public TokenKey build() {
                return new TokenKey(this.token, this.path, this.method, this.authType, this.primerKeyId);
            }

            public String toString() {
                return "PrimerAuthorizationRegistry.TokenKey.TokenKeyBuilder(token=" + this.token + ", path=" + this.path + ", method=" + this.method + ", authType=" + (Object)((Object)this.authType) + ", primerKeyId=" + this.primerKeyId + ")";
            }
        }
    }
}

