# incognito

Different Clojure(Script) serialization protocols like `edn`, `fressian` or
`transit` offer different ways to serialize custom types. In general
they fall back to maps for unknown record types, which is a reasonable
default in many situations. But when you build a distributed data
management system parts of your system might not care about the record
types while others do. This library safely wraps unknown record types
and therefore allows to unwrap them later. It also unifies record
serialization between `fressian` and `transit` as long as you can
express your serialization format in Clojure's default datastructures.

The general idea is that most custom Clojure datatypes (either records
or deftypes) can be expressed in Clojure datastructures if you do not
need a custom binary format, e.g. for efficiency or performance. With
incognito you don't need custom handlers for every serialization
format (but you can still do so, ofc.).

Incognito defaults to a pr-str->read-string roundtrip which is a
reasonable, but inefficient, default for most Clojure types. We use it
for instance to carry the custom type of a [datascript
db](https://github.com/tonsky/datascript) in
[topiq](https://github.com/replikativ/topiq). You don't need to
provide a write handler for incognito except for efficiency reasons or
if your type is not pr-strable (in which case you should make it then
first).



## Usage

Add this to your project dependencies:
[![Clojars Project](http://clojars.org/io.replikativ/incognito/latest-version.svg)](http://clojars.org/io.replikativ/incognito)

Exclude all serialization libraries you don't need, e.g. for edn support only:
```clojure
[io.replikativ/incognito "0.2.0" :exclusions [org.clojure/data.fressian com.cognitect/transit-clj]]
```

In general you can control serialization by `write-handlers` and `read-handlers`:

```clojure
(defrecord Bar [a b])

(def write-handlers (atom {user.Bar (fn [bar] ['user.Bar (assoc bar :c "banana")])}))
(def read-handlers (atom {'user.Bar map->Bar}))
```
*NOTE*: The syntax quote for the read handler which is necessary so you
can deserialize unknown classes.

A write-handler has to return an associative datastructure which is
internally stored as an untyped map together with the tag information.

Extracted from the tests:

### edn

```clojure
(require '[incognito.edn :refer [read-string-safe]])

(let [bar (map->Bar {:a [1 2 3] :b {:c "Fooos"}})]
  (= bar (->> bar
              pr-str
              (read-string-safe {})
              pr-str
              (read-string-safe read-handlers))))
```

For dashed namespace names you need a custom printer to be
ClojureScript conform.

```clojure
(defmethod print-method some_namespace.Bar [v ^java.io.Writer w]
  (.write w (str "#some-namespace.Bar" (into {} v))))
```


### transit
```clojure
(require '[incognito.transit :refer [incognito-write-handler incognito-read-handler]]
         '[cognitect.transit :as transit])

(let [bar (map->Bar {:a [1 2 3] :b {:c "Fooos"}})]
  (= (assoc bar :c "banana")
     (with-open [baos (ByteArrayOutputStream.)]
       (let [writer (transit/writer baos :json
                                    {:handlers {java.util.Map
                                                (incognito-write-handler
                                                 write-handlers)}})]
         (transit/write writer bar)
         (let [bais (ByteArrayInputStream. (.toByteArray baos))
               reader (transit/reader bais :json
                                      {:handlers {"incognito"
                                                  (incognito-read-handler read-handlers)}})]
           (transit/read reader))))))
```

### fressian

```clojure
(require '[clojure.data.fressian :as fress]
         '[incognito.fressian :refer [incognito-read-handlers
                                      incognito-write-handlers]])

(let [bar (map->Bar {:a [1 2 3] :b {:c "Fooos"}})]
  (= (assoc bar :c "banana")
     (with-open [baos (ByteArrayOutputStream.)]
       (let [w (fress/create-writer baos
                                    :handlers
                                    (-> (merge fress/clojure-write-handlers
                                               (incognito-write-handlers write-handlers))
                                        fress/associative-lookup
                                        fress/inheritance-lookup))] ;
         (fress/write-object w bar)
         (let [bais (ByteArrayInputStream. (.toByteArray baos))]
           (fress/read bais
                       :handlers
                       (-> (merge fress/clojure-read-handlers
                                  (incognito-read-handlers read-handlers))
                           fress/associative-lookup)))))))
```

## License

Copyright © 2015-2016 Christian Weilbach

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
