(ns lambdaconnect-graphql.filter 
  (:require [failjure.core :as f]
            [lambdaconnect-graphql.sort :as sort]
            [lambdaconnect-graphql.types :as types]
            [lambdaconnect-graphql.utils :as u]))

(def filter-rules-description
  "Map describing all legal rules.
   Keys are strings that can be used in GraphQL query (field \"comparator\").
   Values are maps describing rules. Possible keys are:
   single-final-value? - if true, filter field must refer to a single value (attribute or aggregate value);
                         otherwise, filter field must refer to set of values
   required-type? - if true, filter field must have type satisfying predicate
   value-required? - if true, value must be supplied in filter
   fn - Function executing filter.
        If value-required? is true, it is a function taking a parsed value from query
           and returning a predicate as if value-required? is false
        If value-required? is false, it is a predicate taking a value (or set of values) generated by db query."
  {">="
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (>= (compare x value) 0))))}
   ">"
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (> (compare x value) 0))))}
   "<="
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (<= (compare x value) 0))))}
   "<"
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (< (compare x value) 0))))}
   "="
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (= (compare x value) 0))))}
   "!="
   {:value-required? true
    :single-final-value? true
    :fn (fn [value] (fn [x] (and (some? x) (not= (compare x value) 0))))}
   "isnull"
   {:value-required? false
    :single-final-value? true
    :fn nil?}
   "notnull"
   {:value-required? false
    :single-final-value? true
    :fn some?}
   "contains"
   {:value-required? true
    :single-final-value? false
    :fn (fn [value] (fn [xs] (boolean (xs value))))}
   "regex"
   {:value-required? true
    :single-final-value? true
    :required-type? #{:db.type/string}
    :fn (fn [value]
          (let [regex (re-pattern value)]
            (fn [x] (and (some? x) (boolean (re-matches regex x))))))}})

(defn parse-filter-subquery
  "Parse a filter object.
   Input:
   entities-by-name, entity-name, db, scoped-ids, rules
   filter-object - as described in restaurant-advisors.graphql.custom.schema:
     * comparator - string, key defined in filter-rules-description
     * value - value to be compared to (as string - this function takes care of parsing that value),
     * field - string describing (possibly) nested relationship.

   Output: failjure error OR function which takes ids of objects to be filtered and returns filtered ids as a set."
  [config entity-name db scoped-ids rules {:keys [negate comparator value field]}]
  (f/attempt-all
   [filter-description (get filter-rules-description comparator
                            (f/fail "Invalid filter comparator %s." comparator))
    parsed-extract (sort/nested-relation->fn config entity-name field (:single-final-value? filter-description)
                                             db scoped-ids rules)
    _ (when-let [required-type? (:required-type? filter-description)]
        (when-not (required-type? (:value-type parsed-extract))
          (f/fail "Invalid type %s of value for filter %s." ((:value-type parsed-extract) types/types-map) comparator)))
    parsed-value (when (:value-required? filter-description)
                   (try ((u/field-parser (:value-type parsed-extract)) value)
                        (catch Exception _
                          (f/fail "Could not parse filter value %s." value))))
    filter-fn (let [f (:fn filter-description)]
                (if (:value-required? filter-description) (f parsed-value) f))]
   (fn [ids] (->> ids
                  ((:result-fn parsed-extract))
                  (keep (fn [[id value]] (when (if negate
                                                 (not (filter-fn value))
                                                 (filter-fn value)) id)))
                  set))))

(defn parse-or-filter
  "Parse a list of or filters."
  [config entity-name db scoped-ids rules filter-or-list]
  (if (empty? (:or filter-or-list))
    (f/fail "Empty or filter supplied (this is most likely not intentional, since empty or equals false).")
    (u/map-failures #(parse-filter-subquery config entity-name db scoped-ids rules %) (:or filter-or-list))))
