(ns lambdaconnect-graphql.patterns
  (:require [clojure.set :as set]
            [clojure.string :as str]
            [failjure.core :as f]
            [lambdaconnect-graphql.filter :as filter]
            [lambdaconnect-graphql.sort :as sort]
            [lambdaconnect-graphql.utils :as u]))

(defn conjv
  [coll v]
  (if (nil? coll)
    (vector v)
    (conj coll v)))

(defn- intov
  [coll v]
  (if (nil? coll)
    v
    (into coll v)))

(defn- to-field-name
  "Identifies the qualified field name for a selection node.  May return nil
  for meta-fields such as __typename."
  [node]
  (get-in node [:field-definition :qualified-name]))

(defn build-selections-map
  "Redefining com.walmartlabs.lacinia.executor/build-selections-map to include directives in returned map.
   In-built @skip and @include directives' behaviour is still unchanged and they will not appear in the map.
   
   TO CONSIDER: This redefines function from a particular version of lacinia library.
   How to make this redefinition independent from lacinia library version?"
  [parsed-query selections {:keys [selection-kind] :as config}]
  (reduce (fn [m selection]
            (if (:disabled? selection)
              m
              (case (selection-kind selection)
  
                :field
                ;; to-field-name returns nil for pseudo-fields, which are skipped
                (if-some [field-name (to-field-name selection)]
                  (let [{:keys [alias selections directives]
                         simple-field-name :field-name} selection
                        arguments (:arguments selection)
                        selections-map (build-selections-map parsed-query selections config)
                        nested-map (cond-> nil
                                     (not (= simple-field-name alias)) (assoc :alias alias)
                                     (seq arguments) (assoc :args arguments)
                                     (seq selections-map) (assoc :selections selections-map)
                                     ;; The change is here!
                                     (seq directives) (assoc :directives (->> directives
                                                                              (map (juxt :directive-name :arguments))
                                                                              (into {}))))]
                    (update m field-name conjv nested-map))
                  m)
  
                :inline-fragment
                (merge-with intov m (build-selections-map parsed-query (:selections selection) config))
  
                :fragment-spread
                (let [{:keys [fragment-name]} selection
                      fragment-selections (get-in parsed-query [:fragments fragment-name :selections])]
                  (merge-with intov m (build-selections-map parsed-query fragment-selections config))))))
          {}
          selections))

(defn app-attr
  [graphql-attr]
  (keyword "app" (name graphql-attr)))

(defn backward-attr
  "Backward attributes appear in autogenerated part of GraphQL schema (in entities-by-name :relationships),
   but not in Datomic schema (:datomic-relationships)."
  [{:keys [entities-by-name custom-resolvers]} graphql-attr]
  (when (not (custom-resolvers graphql-attr))
    (when-let [{:keys [inverse-name inverse-entity]}
               (get-in entities-by-name [(namespace graphql-attr) :relationships (name graphql-attr)])]
      (keyword inverse-entity (str "_" inverse-name)))))


(defn selections-tree
  [context {:keys [parsed-query-key selection-key] :as config}]
  (let [parsed-query (get context parsed-query-key)
        selection (get context selection-key)]
    (build-selections-map parsed-query (:selections selection) config)))

(defn pull-pattern
  "Given a Lacinia request context, return a pull pattern that can be used
   to return all required values from Datomic database in one query."
  ([context config]
   (pull-pattern context config (selections-tree context config)))
  ([context
    {:keys [custom-attributes datomic-attrs] :as config}
    selection-tree]
   (let [;; is selection-tree equal at the top level to a _list_output GraphQL object?
         keys (keys selection-tree)
         namespaces (set (map namespace keys))
         names (set (map name keys))
         is-list-output? (and (= (count namespaces) 1)
                              (str/ends-with? (first namespaces) "_list_output")
                              (or (= #{"value" "count"} names)
                                  (= #{"value"} names)))]
     (if is-list-output?
       (pull-pattern context (->> selection-tree
                                  (keep (fn [[k [v]]]
                                          (when (= (name k) "value")
                                            (:selections v))))
                                  first))
       (let [sel (fn [[graphql-attr [selection]]]
                   (when-let [attr (or (when (datomic-attrs graphql-attr)
                                         graphql-attr)
                                       (let [special-attr (app-attr graphql-attr)]
                                         (when (datomic-attrs special-attr)
                                           special-attr))
                                       (when (contains? custom-attributes graphql-attr)
                                         graphql-attr)
                                       (when-let [backward-attr (backward-attr config graphql-attr)]
                                         [backward-attr :as graphql-attr]))]
                     (let [directives (:directives selection)]
                       (if selection
                         (cond-> {attr (pull-pattern context config (:selections selection))}
                           directives (assoc :directives directives))
                         attr))))
             result (conj (keep sel selection-tree) :db/id)]
         result)))))

(defn top-keys-covered?
  "The context is as follows:

   * Lacinia receives a request to return an object with deeply nested fields.

   * The top level resolver returns a value that, hopefully, due to nested pattern
     produced by function pull-pattern contains all deeply nested fields.
     This might not be true if the request has some fields which have to be computed because
     they are not corresponding to the fields in database (for example, categories of ingredients).

   * However, Lacinia doesn't care and still runs resolver for all nested objects (one at each level).

   This function checks if the initial resolver returned all required keys
   and nested resolver can be skipped."
  [context {:keys [datomic-attrs] :as config} args]
  (cond
    (nil? args)
    true

    (map? args)
    (let [existing-keys (set (map name (keys args)))
          expected-keys (set (keep (fn [graphql-attr]
                                     (when (or (datomic-attrs graphql-attr)
                                               (datomic-attrs (app-attr graphql-attr))
                                               (backward-attr config graphql-attr))
                                       (name graphql-attr)))
                                   (keys (selections-tree context config))))]
      (set/subset? expected-keys existing-keys))

    (sequential? args)
    (or (empty? args)
        (top-keys-covered? context config (first args)))))


(defn get-entity-id
  [entity-name]
  (keyword entity-name (str (gensym))))

(defn sequential-pull-structure
  "Input:
   context - lacinia context
   Output:
    {:RAEntity-id pull-patterns
     :RAEntity-id2 pull-patterns2}
    RAEntity-id - :RAEntityName/gensym
    pull-patterns - [{:relationships [[:RAChildEntity-id pull-pattern] [:RAChildEntity-id1 pull-pattern1]]
                     :attributes [attrs]}
                    {:relationships [[:RAChildEntity-id custom-relationship-key]]
                     :attributes [attrs]}]
                    first map desribes patterns of normal attributes and relationships the second one
                    custom ones"
  ([context {:keys [entities-by-name] :as config}]
   (let [selection-tree (selections-tree context config)]
     (when (seq selection-tree)
       (let [entity-name (if (contains? entities-by-name (namespace (first (keys selection-tree))))
                           (namespace (first (keys selection-tree)))
                           (first (str/split (namespace (first (keys selection-tree))) #"_" 2)))]
         (sequential-pull-structure selection-tree config (get-entity-id entity-name) [] {})))))
  ([selection-tree {:keys [entities-by-name custom-attributes custom-relationships datomic-attrs-only datomic-relationships-only] :as config}
    entity-id main-execution-order main-pull-structure]
   (let [selection-tree-keys (keys selection-tree)
         namespaces (set (map namespace selection-tree-keys))
         names (set (map name selection-tree-keys))
         list-output-value-present? (and (= (count namespaces) 1)
                                         (str/ends-with? (first namespaces) "_list_output")
                                         (or (= #{"value" "count"} names)
                                             (= #{"value"} names)))

         list-output-only-count? (and (= (count namespaces) 1)
                                      (str/ends-with? (first namespaces) "_list_output")
                                      (= #{"count"} names))]

     (cond
       list-output-value-present? (let [entity-name (first (str/split (namespace (first selection-tree-keys)) #"_" 2))
                                        entity-id (if entity-id
                                                    entity-id
                                                    (get-entity-id entity-name))]
                                    (sequential-pull-structure (->> selection-tree
                                                                    (keep (fn [[k [v]]]
                                                                            (when (= (name k) "value")
                                                                              (:selections v))))
                                                                    first)
                                                               config
                                                               entity-id
                                                               main-execution-order
                                                               main-pull-structure))

       list-output-only-count? (let [entity-name (first (str/split (namespace (first selection-tree-keys)) #"_" 2))
                                     entity-id (if entity-id
                                                 entity-id
                                                 (get-entity-id entity-name))]
                                 [{entity-id [{:relationships []
                                               :attributes [:db/id]}
                                              {:relationships []
                                               :attributes []}]}
                                  [{:entity-id entity-id}]])


       :else (reduce (fn [[pull-strucutre execution-order] [graphql-sel-key [graphql-sel-val]]]
                       (when-not entity-id (throw (Exception. "entity-id is nil")))
                       (let [pull-strucutre
                             (if (not (entity-id pull-strucutre))
                               (assoc pull-strucutre entity-id [{:relationships []
                                                                 :attributes [:db/id]}
                                                                {:relationships []
                                                                 :attributes []}])
                               pull-strucutre)
                             execution-order (if (not (some (comp #{entity-id} :entity-id) execution-order))
                                               (conj execution-order {:entity-id entity-id})
                                               execution-order)

                             update-my-pull (fn [val & {:keys [attr? rel? custom?]}]
                                              (when (or (and (not attr?) (not rel?)) (and attr? rel?))
                                                (throw (Exception. (str "Exactly one of :attr? and :rel? must be true"))))
                                              (let [update-pattern (fn [pattern-map value-to-insert]
                                                                     (if rel?
                                                                       (update pattern-map :relationships conj value-to-insert)
                                                                       (update pattern-map :attributes conj value-to-insert)))]
                                                (if (not custom?)
                                                  (update pull-strucutre entity-id
                                                          (fn [[normal-pattern custom-pattern] new-attr]
                                                            [(update-pattern normal-pattern new-attr) custom-pattern])
                                                          val)
                                                  (update pull-strucutre entity-id
                                                          (fn [[normal-pattern custom-pattern] new-attr]
                                                            [normal-pattern (update-pattern custom-pattern new-attr)])
                                                          val))))

                             update-order-with-directives (fn [execution-key [pull-structure execution-order]]
                                                            [pull-structure
                                                             (->> execution-order
                                                                  (mapv (fn [execution-step]
                                                                          (if (= (:entity-id execution-step) execution-key)
                                                                            (if-let [directives (:directives graphql-sel-val)]
                                                                              (assoc execution-step :directives directives)
                                                                              execution-step)
                                                                            execution-step))))])

                             backward-relationship (backward-attr config graphql-sel-key)
                             special-attr (app-attr graphql-sel-key)
                             get-inverse-entity-name (fn [rel-name]
                                                       (-> entities-by-name
                                                           (get (namespace entity-id))
                                                           :relationships
                                                           (get rel-name)
                                                           :inverse-entity))
                             [updated-pull-structure execution-order]
                             (cond
                               (datomic-attrs-only graphql-sel-key)
                               [(update-my-pull graphql-sel-key :attr? true) execution-order]

                               (datomic-attrs-only special-attr)
                               [(update-my-pull special-attr :attr? true) execution-order]

                               (custom-attributes graphql-sel-key)
                               [(update-my-pull graphql-sel-key :attr? true :custom? true) execution-order]

                               (datomic-relationships-only graphql-sel-key)
                               (let [child-id (get-entity-id (get-inverse-entity-name (name graphql-sel-key)))]
                                 (->> (update-my-pull [child-id {graphql-sel-key [:db/id]}] :rel? true)
                                      (sequential-pull-structure (:selections graphql-sel-val) config child-id execution-order)
                                      (update-order-with-directives child-id)))

                               backward-relationship
                               (let [child-id (get-entity-id (namespace backward-relationship))]
                                 (->> (update-my-pull [child-id {[backward-relationship :as graphql-sel-key] [:db/id]}] :rel? true)
                                      (sequential-pull-structure (:selections graphql-sel-val) config child-id execution-order)
                                      (update-order-with-directives child-id)))

                               (custom-relationships graphql-sel-key)
                               (let [child-id (-> graphql-sel-key
                                                  custom-relationships
                                                  :inverse-entity
                                                  get-entity-id)]
                                 (->> (update-my-pull [child-id graphql-sel-key] :rel? true :custom? true)
                                      (sequential-pull-structure (:selections graphql-sel-val) config child-id execution-order)
                                      (update-order-with-directives child-id)))

                               :else
                               (throw (Exception. (str "Unknown graphql-sel-key: " graphql-sel-key))))]
                         [updated-pull-structure execution-order]))
                     [main-pull-structure main-execution-order]
                     selection-tree)))))

(defn parse-filter
  [config entity-name query-filter db scoped-ids rules]
  (some->> query-filter :and seq (u/map-failures #(filter/parse-or-filter config entity-name db scoped-ids rules %))))

(defn parse-sorts
  [config entity-name query-sorts db scoped-ids rules]
  (some->> query-sorts
           seq
           (u/map-failures
            (fn [query-sort]
              (f/attempt-all
               [sorting-options (set (:options query-sort))
                _forbidden-option-combinations (sort/check-forbidden-options-combinations sorting-options)
                parsed-sort (sort/nested-relation->fn
                             config
                             entity-name
                             (:field query-sort)
                             true
                             db
                             scoped-ids
                             rules)]
               (assoc parsed-sort :sorting-options sorting-options))))))

(defn validate-pagination
  [query-pagination]
  (when query-pagination
    (u/fail-many [(when (< (:page query-pagination) 0)
                    "Pages are indexed from 0.")
                  (when (< (:perPage query-pagination) 0)
                    "Page must have at least 0 objects.")])))

(defn validate-and-parse-seq-pull-structure
  [seq-pull-structure config db scoped-ids rules]
  (u/update-in-failure
   seq-pull-structure
   [1]
   (fn [execution-steps]
     (u/map-failures
      (fn [execution-step]
        (let [entity-name (-> execution-step :entity-id namespace)
              {:keys [filter sort pagination]} (-> execution-step :directives :filterSortPaginate)]
          (f/attempt-all [parsed-filter (parse-filter config entity-name filter db scoped-ids rules)
                          parsed-sorts (parse-sorts config entity-name sort db scoped-ids rules)
                          _ (validate-pagination pagination)]
                         (assoc execution-step
                                :parsed-filter parsed-filter
                                :parsed-sorts parsed-sorts
                                :pagination pagination))))
      execution-steps))))

