(ns lambdaconnect-graphql.pull
  (:require [clojure.set :as set]
            [failjure.core :as f]
            [lambdaconnect-graphql.convert :as c]
            [lambdaconnect-graphql.patterns :as patterns]
            [lambdaconnect-graphql.scope :as scope]
            [lambdaconnect-graphql.sort :as sort]
            [lambdaconnect-graphql.utils :as u]))

(defn deref-or-throw
  [promise ex-msg]
  (let [deref-timeout 10000 ;; 10 seconds
        ans (deref promise deref-timeout :timeout)]
    (if (= :timeout ans)
      (throw (Exception. ex-msg))
      ans)))

(defn take-page
  [entities {:keys [page perPage] :as pagination}]
  (if pagination
    (->> entities (drop (* page perPage)) (take perPage))
    entities))

(defn pull-entities
  "Input:
   entity-name - name of the entity which structure belongs to (string)
   entity-structure - structure generated by sequential-pull-structure
   uuids - a list of uuids to pull. If nil, all entities will be pulled
   Output:
   A tuple of queries for \"normal\" and custom attributres and relationships
   For example queries look into test/restaurant_advisors/test_db_sequential_data_pull/get-pull-entity-query-test"
  [{:keys [custom-attributes custom-relationships entities-by-name q]} entity-name [normal-pull custom-pull] ids user-scoped-ids db user-rules]
  (let [normal-pull-pattern (into (:attributes normal-pull) (map second (:relationships normal-pull)))
        normal-query `[:find (~'pull ~'?db-id ~normal-pull-pattern)
                       :in ~'$ ~'[?db-id ...]
                       :where ~'[?db-id :app/uuid]]
        relationships-info (->> normal-pull
                                :relationships
                                (map (fn [[_ sub-pull-pattern]]
                                       (let [pattern-key (if (keyword? sub-pull-pattern) sub-pull-pattern (ffirst sub-pull-pattern))
                                             relationship (if (vector? pattern-key) ;; [... :as ...]
                                                            (last pattern-key)
                                                            pattern-key)
                                             to-many? (if-let [custom-relationship (custom-relationships relationship)]
                                                        (:to-many custom-relationship)
                                                        (get-in entities-by-name [entity-name :relationships (name relationship) :to-many]))]
                                         [relationship to-many?]))))
        normal-pulled-entities (->> (q normal-query db ids)
                                    (map (fn [[entity]]
                                           [(:db/id entity)
                                            (let [updated-entity
                                                  (reduce (fn [entity [relationship to-many]]
                                                            (if (contains? entity relationship)
                                                              (update entity relationship
                                                                      #(as-> % value
                                                                         (if (and (not to-many) (vector? value)) (first value) value)
                                                                         (if to-many (set (map :db/id value)) (:db/id value))))
                                                              entity))
                                                          entity
                                                          relationships-info)
                                                  updated-entity
                                                  (reduce (fn [entity attribute]
                                                            (if (contains? entity attribute)
                                                              entity
                                                              (assoc entity attribute nil)))
                                                          updated-entity
                                                          (:attributes normal-pull))]
                                              updated-entity)]))
                                    (into {}))
        ids (set (keys normal-pulled-entities))
        custom-attributes (->> custom-pull
                               :attributes
                               (map (fn [attribute]
                                      (let [attr-fn (-> custom-attributes attribute :fn)
                                            id->attr (attr-fn {:db db :ids ids :rules user-rules :scoped-ids user-scoped-ids})]
                                        (->> ids
                                             (map (fn [id] [id {attribute (id->attr id)}]))
                                             (into {})))))
                               (apply merge-with merge))

        custom-relationships (->> custom-pull
                                  :relationships
                                  (map (fn [[_ relationship]]
                                         (let [attr-fn (-> custom-relationships relationship :fn)
                                               id->attr (attr-fn {:db db :ids ids :rules user-rules :scoped-ids user-scoped-ids})]
                                           (->> ids
                                                (map (fn [id] [id {relationship (id->attr id)}]))
                                                (into {})))))
                                  (apply merge-with merge))]
    (merge-with merge normal-pulled-entities custom-attributes custom-relationships)))

(defn perform-single-level-query
  "Filter and sort act in query, but pagination acts in parent's query.
   As an example for why's this, consider a query where we ask for main entities M' related entities R
   and to get Rs, we use some filter, some sort and pagination {page: 0, perPage: 2}.
   First level query returns two Ms: m1 with Rs 1, 4, 2 and m2 with Rs 3, 5, 2.
   Those ids are grouped before passing to second level query.
   Second level query queries for Rs 1, 4, 2, 3, 5.
   Suppose that it returns r1, r4, r2, r3, r5 before filtering,
   the filter discards r1 and sort sets them in order r2, r3, r4, r5.
   Those 4 Rs are returned in order to M query, but we have to do pagination separately for each recipe,
   we are supposed to return m1 with r2, r4 and m2 with r2, r3.
   The result at the top level is not paginated, it is supposed to be paginated outside of this file,
   as output result requires counting answers from all pages."
  [{:keys [q entities-by-name custom-relationships force-sequential-pull-execution?] :as config} entity-id execution-steps-map pull-structure promises-map db user-scoped-ids user-rules]
  (let [execution-step (entity-id execution-steps-map)
        input-ids @(-> promises-map entity-id :input)
        entity-name (namespace entity-id)
        scoped-ids @(user-scoped-ids entity-name)
        input-ids (if (= :all input-ids) (vec scoped-ids) (filter scoped-ids input-ids))
        active-ids-set (set (map first (q '[:find ?id :in $ [?id ...] :where [?id :app/active true]] db input-ids)))
        active-ids (filter active-ids-set input-ids)
  
        filtered-ids (if-let [query-filter (:parsed-filter execution-step)]
                       (let [filter-set (->> query-filter
                                             (pmap (fn [or-filters]
                                                     (->> or-filters
                                                          (pmap #(% active-ids))
                                                          (apply set/union))))
                                             (apply set/intersection (set active-ids)))]
                         (filter filter-set active-ids))
                       active-ids)
  
        sorted-ids (if-let [query-sort (:parsed-sorts execution-step)]
                     (let [sort-values
                           (->> query-sort
                                (pmap #(sort/execute-sort-subquery % filtered-ids))
                                (apply merge-with concat)
                                (map (fn [[value sort-values]] {:value value :sort-value sort-values})))
                           comparators (->> query-sort
                                            (map (fn [{:keys [sorting-options]}]
                                                   (reduce (fn [custom-comparator [option combinator]]
                                                             (cond-> custom-comparator
                                                               (option sorting-options)
                                                               combinator))
                                                           compare
                                                           sort/comparator-combinators))))]
                       (->> sort-values
                            (sort-by :sort-value #(sort/comparator-by-coordinates comparators %1 %2))
                            (mapv :value)))
                     (vec filtered-ids))
  
        pulled-result (pull-entities config entity-name pull-structure sorted-ids user-scoped-ids db user-rules)
        sorted-entities (mapv pulled-result sorted-ids)
  
        subqueries-data
        (->> pull-structure
             (map :relationships)
             (apply concat)
             (mapv (fn [[sub-entity-id sub-pull-pattern]]
                     (let [pattern-key (if (keyword? sub-pull-pattern) sub-pull-pattern (ffirst sub-pull-pattern))
                           relationship (if (vector? pattern-key) ;; [... :as ...]
                                          (last pattern-key)
                                          pattern-key)
                           ids (->> sorted-entities (map relationship) u/union)
                           to-many? (if-let [custom-relationship (custom-relationships relationship)]
                                      (:to-many custom-relationship)
                                      (get-in entities-by-name [entity-name :relationships (name relationship) :to-many]))]
                       (deliver (-> promises-map sub-entity-id :input) ids)
                       (when force-sequential-pull-execution?
                         @(-> promises-map sub-entity-id :output))
                       [sub-entity-id relationship to-many?]))))
  
          ;; replace ids of related objects with values, apply pagination
        final-result
        (reduce (fn [result [sub-entity-id relationship-key to-many?]]
                  (let [subquery-result (-> promises-map sub-entity-id :output (deref-or-throw (str "Timeout in query " sub-entity-id)))
                        subquery-result-map (->> subquery-result (map (juxt :db/id identity)) (into {}))
                        update-result-entity
                        (fn [entity]
                          (let [new-entity (update entity relationship-key
                                                   (fn [relationship-ids]
                                                     (cond (number? relationship-ids)
                                                           (subquery-result-map relationship-ids)
  
                                                           (set? relationship-ids)
                                                           (let [selected-entities (filter (comp relationship-ids :db/id) subquery-result)
                                                                 paginated-entities (->> execution-steps-map sub-entity-id :pagination (take-page selected-entities))]
                                                             (vec paginated-entities))
  
                                                           :else (when to-many? []))))]
                            new-entity))
                        ans (mapv update-result-entity result)]
                    ans))
                sorted-entities
                subqueries-data)]
    (deliver (-> promises-map entity-id :output) final-result)))

(defn perform-sequential-pull
  [{:keys [q] :as config} context db [pull-structures execution-order] user-rules user-db-id uuids]
  (let [promises-map (->> pull-structures
                          (map (fn [[k]] {k {:input (promise) :output (promise)}}))
                          (into {}))
        user-scoped-ids (scope/scoped-ids config context user-db-id db user-rules)
        execution-steps-map (->> execution-order
                                 (map (juxt :entity-id identity))
                                 (into {}))
        ids (if uuids
              (->> (q '[:find ?uuid ?id
                        :in $ [?uuid ...]
                        :where [?id :app/uuid ?uuid]]
                      db uuids)
                   (into {})
                   (#(map % uuids)))
              :all)
        init-key (-> execution-order first :entity-id)]
    (deliver (-> promises-map init-key :input) ids)
    (doseq [{:keys [entity-id]} execution-order]
      (future (perform-single-level-query config entity-id execution-steps-map (entity-id pull-structures) promises-map db user-scoped-ids user-rules)))
    (-> promises-map init-key :output (deref-or-throw (str "Timeout in query " init-key)))))

(defn pull-by-uuids
  [{:keys [user-rules-by-id] :as config} context db uuids user-db-id seq-pull-structure]
  (let [uuids (map u/string->uuid uuids)]
    (if (seq uuids)
      (f/attempt-all
       [rules (user-rules-by-id context user-db-id db)
        scoped-ids (scope/scoped-ids config context user-db-id db rules)
        seq-pull-structure (patterns/validate-and-parse-seq-pull-structure seq-pull-structure config db scoped-ids rules)]
       (->> (perform-sequential-pull config context db seq-pull-structure rules user-db-id uuids)
            (c/db-convert config))
       (f/when-failed [e] (u/failure->lacinia-error config e)))
      [])))

(defn pull-all
  "Output:
   Pulls all entities matching pattern given from db. Resulting entities are filtered,
   sorted or paginated with use of the objects defined in restaurant-advisors.graphql.custom.schema."
  [db {:keys [user-rules-by-id] :as config} context _entity-name user-db-id seq-pull-structure
   & {:keys [query-sorts query-filter query-pagination]}]
  (if seq-pull-structure
    (f/attempt-all
     [rules (user-rules-by-id context user-db-id db)
      scoped-ids (scope/scoped-ids config context user-db-id db rules)
      seq-pull-structure (some-> seq-pull-structure
                                 (assoc-in [1 0 ;; first execution step
                                            :directives :filterSortPaginate] {:filter query-filter
                                                                              :sort query-sorts
                                                                              :pagination query-pagination})
                                 (patterns/validate-and-parse-seq-pull-structure config db scoped-ids rules))

      unpaginated-final-answer (perform-sequential-pull config context db seq-pull-structure rules user-db-id nil)

      paginated-final-answer (take-page unpaginated-final-answer query-pagination)]
     {:value (c/db-convert config paginated-final-answer)
      :count (count unpaginated-final-answer)}
     (f/when-failed [e] (u/failure->lacinia-error config e)))
    {:value []}))


(defn pull-ref
  [{:keys [q] :as config} db id pattern relation]
  (->> (q '[:find (pull ?ref-id pattern)
            :in $ pattern ?id ?relation
            :where [?id ?relation ?ref-id]] db pattern id relation)
       ffirst
       (c/db-convert config)))

(defn pull-refs
  [{:keys [q] :as config} db id pattern relation]
  (->> (q '[:find (pull ?ref-id pattern)
            :in $ pattern ?id ?relation
            :where [?id ?relation ?ref-id]] db pattern id relation)
       (map first)
       (c/db-convert config)))

(defn reverse-pull-ref
  [{:keys [pull] :as config} db id pattern backward-relation]
  (->> (pull db [{backward-relation pattern}] id)
       first
       second
       (c/db-convert config)))

(defn reverse-pull-refs
  [{:keys [pull] :as config} db id pattern backward-relation]
  (->> (pull db [{backward-relation pattern}] id)
       first
       second
       (c/db-convert config)))
