(ns component.keycloak
  (:require [integrant.core :as ig]
            [medley.core :refer [deep-merge]]
            [malli.core :as m]
            [clojure.tools.logging :as log]
            [lib.security.oidc.jwt :as jwt]
            [lib.security.oidc.discovery :as discovery]))

(defn fetch-keyset!
  {:added "0.1.1"}
  [endpoint]
  (log/infof "fetching jwks from %s" endpoint)
  (-> endpoint
      discovery/issuer->config-uri
      discovery/get-openid-config
      (get "jwks_uri")
      jwt/get-keyset))
;; ___________                                .___
;; \_   _____/__  ______________    ____    __| _/
;;  |    __)_\  \/  /\____ \__  \  /    \  / __ |
;;  |        \>    < |  |_> > __ \|   |  \/ /_/ |
;; /_______  /__/\_ \|   __(____  /___|  /\____ |
;;         \/      \/|__|       \/     \/      \/

(defmethod ig/expand-key :component/keycloak
  [k v]
  {k (deep-merge {} v)})

;;    _____                               __
;;   /  _  \   ______ ______ ____________/  |_
;;  /  /_\  \ /  ___//  ___// __ \_  __ \   __\
;; /    |    \\___ \ \___ \\  ___/|  | \/|  |
;; \____|__  /____  >____  >\___  >__|   |__|
;;         \/     \/     \/     \/

(def spec
  [:map
   [:host {:title "Keycloak host endpoint"} string?]
   [:realm {:title "Keycloak realm"} string?]])

(defmethod ig/assert-key :component/keycloak
  [_ system]
  (assert (m/validate spec system)
          (m/explain spec system)))

;; .___       .__  __
;; |   | ____ |__|/  |_
;; |   |/    \|  \   __\
;; |   |   |  \  ||  |
;; |___|___|  /__||__|
;;          \/

(defmethod ig/init-key :component/keycloak
  [_ {:keys [host realm] :as system}]
  (let [endpoint (str host "/realms/" realm)
        keyset (atom (fetch-keyset! endpoint))]
    (log/infof "starting keycloak component on %s" endpoint)
    (assoc system
           :endpoint endpoint
           :keyset keyset)))

;;   ___ ___        .__   __
;;  /   |   \_____  |  |_/  |_
;; /    ~    \__  \ |  |\   __\
;; \    Y    // __ \|  |_|  |
;;  \___|_  /(____  /____/__|
;;        \/      \/

(defmethod ig/halt-key! :component/keycloak
  [_ {:keys [] :as system}]
  (log/info "stopping keycloak component"))
