(ns lib.sentry.api
  (:require [clojure.string :as string]
            [jsonista.core :as json]
            [clj-http.client :as http])
  (:import (java.net InetAddress)
           (java.util Date)))

(defn- make-sentry-url
  [uri project-id]
  (format "%s/api/%s/store/"
          uri project-id))

(def sentry-client "io.trosa/component.sentry")

(defn- make-sentry-header
  [ts key secret]
  (format "Sentry sentry_version=2.0, sentry_client=%s, sentry_timestamp=%s, sentry_key=%s, sentry_secret=%s"
          sentry-client ts key secret))

(defn- send-packet
  [{:keys [key secret project_id uri]} packet-info]
  (let [url (make-sentry-url uri project_id)
        timestamp (str (.getTime (Date.)))
        header (make-sentry-header timestamp key secret)]
    (http/post url
               {:insecure?        true
                :throw-exceptions false
                :headers          {"X-Sentry-Auth" header
                                   "User-Agent"    sentry-client}
                :body             (json/write-value-as-string packet-info)})))

(defn- parse-dsn
  [dsn]
  (let [[proto-auth url] (string/split dsn #"@")
        [protocol auth] (string/split proto-auth #"://")
        [key secret] (string/split auth #":")]
    {:key        key
     :secret     secret
     :uri        (format "%s://%s" protocol
                         (string/join
                          "/" (butlast (string/split url #"/"))))
     :project_id (Integer/parseInt (last (string/split url #"/")))}))

(def localhost
  (memoize
   (fn []
     (.getHostName (InetAddress/getLocalHost)))))

(defn capture
  "Capture a sentry event and send it to the specified dsn (Data Source Name).
   dsn should match the following: `{PROTOCOL}://{PUBLIC_KEY}:{SECRET_KEY}@{HOST}{PATH}/{PROJECT_ID}`
   event-info arbitrary map sent as sentry event"
  [dsn event-info]
  (send-packet
   (parse-dsn dsn)
   (merge
    {:level       "error"
     :platform    "clojure"
     :server_name (localhost)}
    event-info)))
