(ns lib.telemetry.instrumentation
  (:require [clojure.string :as str])
  (:import io.opentelemetry.semconv.ClientAttributes
           io.opentelemetry.semconv.ErrorAttributes
           io.opentelemetry.semconv.HttpAttributes
           io.opentelemetry.semconv.ServerAttributes
           io.opentelemetry.semconv.UrlAttributes
           io.opentelemetry.semconv.UserAgentAttributes
           io.opentelemetry.semconv.incubating.HttpIncubatingAttributes))

;; .___                 __                                      __
;; |   | ____   _______/  |________ __ __  _____   ____   _____/  |_  ______
;; |   |/    \ /  ___/\   __\_  __ \  |  \/     \_/ __ \ /    \   __\/  ___/
;; |   |   |  \\___ \  |  |  |  | \/  |  /  Y Y  \  ___/|   |  \  |  \___ \
;; |___|___|  /____  > |__|  |__|  |____/|__|_|  /\___  >___|  /__| /____  >
;;          \/     \/                          \/     \/     \/          \/

(defn- pedestal-merge-headers-attrs
  [init prefix captured-headers headers]
  (persistent!
   (reduce (fn [m header-name]
             (if-let [v (get headers header-name)]
               (assoc! m (str prefix (str/replace header-name \- \_)) [v])
               m))
           (transient init)
           captured-headers)))

(defn- pedestal-client-ip
  [forwarded x-forwarded-for remote-addr]
  (or (some->> forwarded
               (re-find #"(?i)for=([^,;]*)")
               second
               str/trim
               not-empty)
      (some->> x-forwarded-for
               (re-find #"[^,]*")
               str/trim
               not-empty)
      remote-addr))

(defn pedestal-server-request-attrs
  [request captured-request-headers]
  (let [{:keys [headers request-method scheme uri query-string remote-addr]} request
        {:strs [user-agent content-length host forwarded x-forwarded-for]} headers
        [_ host-name host-port] (re-find #"^(.*?)(?::(\d*))?$" host)]
    (cond-> {HttpAttributes/HTTP_REQUEST_METHOD (str/upper-case (name request-method))
             UrlAttributes/URL_SCHEME        (name scheme)
             ServerAttributes/SERVER_ADDRESS host-name
             UrlAttributes/URL_PATH          uri
             UrlAttributes/URL_QUERY         query-string
             ClientAttributes/CLIENT_ADDRESS (pedestal-client-ip forwarded x-forwarded-for remote-addr)}
      user-agent      (assoc UserAgentAttributes/USER_AGENT_ORIGINAL user-agent)
      content-length  (assoc HttpIncubatingAttributes/HTTP_REQUEST_BODY_SIZE
                             (try
                               (Long/parseLong content-length)
                               (catch Throwable _)))
      (seq host-port) (assoc ServerAttributes/SERVER_PORT
                             (try
                               (Long/parseLong host-port)
                               (catch Exception _)))
      captured-request-headers
      (pedestal-merge-headers-attrs "http.request.header." captured-request-headers headers))))
