(ns lib.webserver.routes
  (:require [clojure.tools.logging :as log]
            [reitit.swagger :as swagger]
            [reitit.coercion.malli]
            [reitit.swagger-ui :as swagger-ui]))

(defn ^:private ping-handler
  "For each webserver resources injected, compute the ping-fn functio
   if exposed by the component and then return a status of all of thems.

   If all are not raising exceptions, return 200 status, otherwise 599."
  {:added "1.1.0"}
  [resources]
  (let [pingable (filter (comp :ping-fn second) resources)]
    (fn [_]
      (let [pings
            (->> pingable
                 (map (fn [[k {ping :ping-fn}]]
                        (try
                          (let [start (System/currentTimeMillis)
                                r (ping)
                                delay (- (System/currentTimeMillis) start)]
                            [k (assoc r :delay delay)])
                          (catch Exception e
                            (log/error "ping" k (.getMessage e))
                            {k {:ok false
                                :message (.getMessage e)}}))))
                 (into {}))
            ok? (every? (comp :ok val) pings)]
        {:status (if ok? 200 599)
         :body
         {:ok         ok?
          :hostname   (.getHostName (java.net.InetAddress/getLocalHost))
          :resources  (keys resources)
          :components pings}}))))

(defn ^:private ping-route
  "Create the ping route"
  {:added "1.1.0"}
  [resources]
  ["/ping" {:name :webserver.core/ping
            :no-metrics true
            :no-tracing true
            :tags ["tooling"]
            :get {:summary "route for health check"
                  :description
                  "health check route returning
                        + `ok`         boolean indicating if everything works fine
                        + `version`    current running version of the app
                        + `app`        app name
                        + `hostname`   ...
                        + `components` a map component statuses containing at least a boolean `ok`"
                  :handler (ping-handler resources)}}])

(defn ^:private metrics-route
  "Create the metrics routes, route name is dynamically generated
   from actual spec."
  {:added "1.1.0"}
  [{:keys [route expose] :as metrics}]
  (when (and metrics route)
    [route {:name :webserver.core/metrics
            :get {:no-doc true
                  :no-metrics true
                  :no-tracing true
                  :resources [:metrics]
                  :handler (fn get-metrics [_]
                             {:status 200
                              :body (expose)})}}]))

(defn swagger-routes
  "Create the swagger route"
  {:added "1.1.0"}
  [{:keys [info]}]
  ["" {:no-doc     true
       :no-tracing true
       :no-metrics true}
   ["/swagger.json" {:get {:swagger {:info info}
                           :handler (swagger/create-swagger-handler)}}]
   ["/swagger/*"    {:get (swagger-ui/create-swagger-ui-handler)}]])

(defn stock-routes
  "The default generated routes. Every webservers have
   at least the /ping route enabled, other one will have
   the behavior differents depending the resource availability status."
  {:added "1.1.0"}
  [{:keys [metrics app-infos] :as resources}]
  [(ping-route     resources)
   (swagger-routes app-infos)
   (metrics-route  metrics)])
