(ns volga-firebird.unit.common.excel-transform
  (:require [axel-f.core :as axel-f]))

(defn- parse-number [number]
  (#?(:clj read-string
      :cljs js/parseFloat)
   number))

(defmulti coerce (fn [s type] type))

(defmethod coerce "string" [s _]
  (str s))

(defmethod coerce "integer" [s _]
  (if (string? s)
    (or (parse-number s) s)
    (int s)))

(defmethod coerce "float" [s _]
  (if (string? s)
    (or (parse-number (str s)) s)
    (float s)))

(defmethod coerce "boolean" [s _]
  (if (string? s)
    (cond
      (= (str s) "true") true
      (= (str s) "false") false
      :otherwise false)
    (boolean s)))

(defmethod coerce :default [s _] s)

(defn- deep-merge [& maps]
  (if (every? map? maps)
    (apply merge-with deep-merge maps)
    (last maps)))

(defn- deep-merge-with [f & maps]
  (apply
   (fn m [& maps]
     (if (every? map? maps)
       (apply merge-with m maps)
       (apply f maps)))
   maps))

(defn- deep-hash-map* [target value type]
  (if (empty? target)
    value
    (if (= (first target) "*")
      (if (empty? (rest target))
        (coerce value type)
        (mapv #(deep-hash-map* (rest target) % type)
              value))
      (assoc {} (keyword (first target))
             (deep-hash-map* (rest target) value type)))))

(defn typed-merge [v1 v2]
  (cond
    (and (map? v1) (map? v2))
    (deep-merge-with typed-merge v1 v2)

    (and (coll? v1) (coll? v2))
    (map #(deep-merge-with typed-merge %1 %2)
         v1
         v2)

    :otherwise
    (throw (ex-info "Wrong merge" {:type ::wrong-merge :left v1 :right v2}))))

(defn- deep-hash-map [& keyvals]
  (apply (partial deep-merge-with typed-merge {})
         (map (fn [[target value type]]
                (deep-hash-map* target value type))
              (partition 3 keyvals))))

(defn ->transform-function [attributes]
  (let [attributes (map (fn [{:keys [formula name] :as attribute}]
                          (let [attr-name (axel-f/compile name)
                                attr-formula (axel-f/compile formula)]
                            (assoc attribute
                                   :formula (cond
                                              (or (string? attr-formula)
                                                  (number? attr-formula))
                                              [attr-formula]

                                              :otherwise
                                              attr-formula)
                                   :name (cond
                                           (= :OBJREF (first attr-name))
                                           (vec (rest attr-name))

                                           (string? attr-name)
                                           [attr-name]

                                           :otherwise
                                           attr-name))))
                        attributes)]
    (fn [request]
      (apply deep-hash-map
             (apply concat
                    (map (fn [{:keys [formula name type]}]
                           [name
                            (axel-f/run formula request)
                            type])
                         attributes))))))
