(ns volga-firebird.unit.request
  (:require [cheshire.core :as cheshire]
            [clj-time.core :as t]
            [clojure.string :as string]
            [reitit.impl :as impl]
            [volga-firebird.unit :as unit]))

(defn- path->hierarchy [path]
  (->> (impl/segments path)
      (filter #(not (empty? %)))
      (reduce (fn [x y]
                (let [p (last x)]
                  (conj x (if p
                            (string/join "/" [p y])
                            y))))
              [])
      (map-indexed #(string/join "/" [%1 %2]))))

(defn- request->context [{{request-accept          "accept"
                           request-accept-encoding "accept-encoding"
                           request-host            "host"
                           request-forwarded-host  "x-forwarded-host"
                           request-forwarded-ip    "x-forwarded-ip"
                           request-user-agent      "user-agent"} :headers
                          request-content-type                   :muuntaja/request
                          request-content-length                 :content-length
                          response-content-type                  :muuntaja/response
                          request-path                           :path-info
                          request-ip                             :remote-addr
                          request-method                         :request-method
                          request-scheme                         :scheme
                          request-body                           :body-params
                          request-query-string                   :query-string
                          {user-id        :id
                           api-key-issuer :issuer
                           api-key-roles  :roles}                :user
                          pipeline-project-id                    :pipeline-project-id}]
  (let [request-host (or request-forwarded-host request-host)
        request-at   (t/now)]
    {:user_id                 user-id
     :api_key_issuer          api-key-issuer
     :api_key_roles           api-key-roles
     :request_accept          request-accept
     :request_accept_encoding request-accept-encoding
     :request_at              request-at
     :request_content_type    (string/join "; " (vals request-content-type))
     :request_content_length  (if (< request-content-length 0)
                                0
                                request-content-length)
     :request_hierarchy       (path->hierarchy request-path)
     :request_host            request-host
     :request_ip              (or request-forwarded-ip request-ip)
     :request_method          (-> request-method name string/upper-case)
     :request_path            request-path
     :request_scheme          (name (or request-scheme :not-available))
     :request_url             (str (name (or request-scheme :not-available))
                                   "://"
                                   request-host
                                   request-path)
     :request_url_query       request-query-string
     :request_user_agent      request-user-agent
     :request_body            (cheshire/generate-string request-body)
     :response_content_type   (string/join "; " (vals response-content-type))
     :response_time           (System/currentTimeMillis)
     :parent_id               nil
     :sub_request             false
     :project_id              pipeline-project-id
     :request_id              (str (java.util.UUID/randomUUID))}))

(defmethod unit/unit->unit-fn :request [_]
  (fn [{request-format :muuntaja/format
       :keys          [request-method headers parameters
                       remote-addr uri]
       :as            request}]
    (merge parameters
           {:method      request-method
            :format      request-format
            :headers     headers
            :remote-addr remote-addr
            :uri         uri
            :context {:request (request->context request)}})))
