(ns volga-firebird.core
  (:require [volga.core :as volga]
            [volga-firebird.unit]
            [volga-firebird.unit.context]
            [volga-firebird.unit.request]
            [volga-firebird.unit.secure-store]
            [volga-firebird.unit.rest]
            [volga-firebird.unit.soap]
            [volga-firebird.unit.transform]
            [volga-firebird.unit.response]
            [cheshire.generate :refer [add-encoder]]
            [clojure.core.async :as async])
  (:import [org.joda.time DateTime]))

(add-encoder DateTime
             (fn [c jsonGenerator]
               (.writeString jsonGenerator (str c))))

(defn prepare-graph [unit]
  (volga/unit->graph unit
                     #'volga-firebird.unit/unit-preprocessor))

(defn prepare-pipeline [graph]
  (volga/graph->pipeline graph
                         #'volga-firebird.unit/unit->unit-fn))

(defn run-pipeline [pipeline request]
  (let [input (get-in pipeline [:request :input-channel])
        output (get-in pipeline [:response :output-channel])]
    (async/go (async/>! input request))
    (async/<!! output)))

(comment

  (def input-xml "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:blz=\"http://thomas-bayer.com/blz/\">
   <soap:Header/>
   <soap:Body>
      <blz:getBank>
         <!--type: string-->
         <blz:blz>?</blz:blz>
      </blz:getBank>
   </soap:Body>
  </soap:Envelope>")

  (def output-xml "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:blz=\"http://thomas-bayer.com/blz/\">
   <soap:Header/>
   <soap:Body>
      <blz:getBankResponse>
         <blz:details>
            <!--Optional:-->
            <!--type: string-->
            <blz:bezeichnung>?</blz:bezeichnung>
            <!--Optional:-->
            <!--type: string-->
            <blz:bic>?</blz:bic>
            <!--Optional:-->
            <!--type: string-->
            <blz:ort>?</blz:ort>
            <!--Optional:-->
            <!--type: string-->
            <blz:plz>?</blz:plz>
         </blz:details>
      </blz:getBankResponse>
   </soap:Body>
  </soap:Envelope>")

  (def fault-xml "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\">
  <!--Optional:-->
  <soap:Header>
    <!--You may enter ANY elements at this point-->
  </soap:Header>
  <soap:Body>
    <!--You may enter ANY elements at this point-->
    <soap:Fault>
      <soap:Code>
        <!--type: faultcodeEnum - enumeration: [soap:DataEncodingUnknown,soap:MustUnderstand,soap:Receiver,soap:Sender,soap:VersionMismatch]-->
        <soap:Value>?</soap:Value>
        <!--Optional:-->
        <soap:Subcode>
          <!--type: QName-->
          <soap:Value>?</soap:Value>
          <!--Optional:-->
          <soap:Subcode/>
        </soap:Subcode>
      </soap:Code>
      <soap:Reason>
        <!--1 or more repetitions:-->
        <soap:Text xml:lang=\"?\">?</soap:Text>
      </soap:Reason>
      <!--Optional:-->
      <!--type: anyURI-->
      <soap:Node>?</soap:Node>
      <!--Optional:-->
      <!--type: anyURI-->
      <soap:Role>?</soap:Role>
      <!--Optional:-->
      <soap:Detail>
        <!--You may enter ANY elements at this point-->
      </soap:Detail>
    </soap:Fault>
  </soap:Body>
  </soap:Envelope>")

  (def unit
    {:name :translations
     :inputs [{:name :ru
               :inputs [:request]
               :config {:type "REST"
                        :url "https://www.jetradar.com/autocomplete/places"
                        :http-method "GET"
                        :parameters {:query [{:name "q"
                                              :formula "request.query.term"}
                                             {:name "locale"
                                              :formula "'ru'"}]}}}

              {:name :soap-blz
               :inputs [{:name :auth-set
                         :inputs [:request]
                         :config {:type "SECURE_STORE"
                                  :data-set {:blz "74061101"}}}
                        :request]
               :config {:type "SOAP"
                        :url "http://www.thomas-bayer.com/axis2/services/BLZService"
                        :parameters {:body [{:name "Envelope.Body.getBank.blz.__value"
                                             :formula "request.body.blz"}]}
                        :soap-version "1.2"
                        :soap-action "getBank"
                        :soap-envelopes {:input-xml input-xml
                                         :output-xml output-xml
                                         :fault-xml fault-xml}}}]
     :config {:type "TRANSFORM"
              :attributes [{:name "body.places[*].type"
                            :formula "ru.body[*].type"}

                           {:name "body.places[*].iata_code"
                            :formula "ru.body[*].code"}

                           {:name "body.places[*].translations.ru"
                            :formula "ru.body[*].name"}

                           {:name "body.soap"
                            :formula "soap-blz.body"}

                           {:name "status"
                            :formula "soap-blz.status"}]}})

  (def pipeline
    (atom
     (-> unit
         (volga/unit->graph #'volga-firebird.unit/unit-preprocessor)
         (volga/graph->pipeline #'volga-firebird.unit/unit->unit-fn))))

  (volga/unit->graph unit #'volga-firebird.unit/unit-preprocessor)

  (defn ^:export push [obj]
    (async/go
      (async/>! (get-in @pipeline [:request :input-channel]) obj)))

  (defprotocol PClosable
    (close [this]))

  (defn ^:export run-handler [callback]
    (let [handler (async/go-loop []
                    (callback
                     (async/<! (get-in @pipeline [:response :output-channel])))
                    (recur))]
      (reify
        PClosable
        (close [this] (async/close! handler)))))

  (defn inspector [{:keys [response]}]
    (clojure.pprint/pprint response))

  (run-handler inspector)

  (push {:path-info "/foo/bar/baz"
         :content-length 100
         :request-method :get
         :parameters {:query {:term "LAX"}
                      :body {:blz "74061101"}}})

  (let [pipeline (-> unit prepare-graph prepare-pipeline) ]
    (run-pipeline pipeline
                  {:path-info "/foo/bar/baz"
                   :content-length 100
                   :request-method :get
                   :parameters {:query {:term "LAX"}
                                :body {:blz "74061101"}}}))


  )
