(ns volga-firebird.unit.response
  (:require [cheshire.core :as cheshire]
            [firebird-logger.core :as logger]
            [volga-firebird.unit :as unit]))

(defn- find-error-status-and-message [errors]
  (->> errors
      (map (fn [[_ v]]
             (when-let [data (ex-data v)]
               [(:status data) (.getMessage v)])))
      (filter #(let [[status message] %]
                 (or status message)))
      first))

(defn- wrap-response [response]
  (let [result (->> response
                   (filter (fn [[k v]]
                             (not (or (= :context k)
                                      (instance? Throwable v)))))
                   (into {}))
        errors (->> response
                   (filter (fn [[_ v]]
                             (instance? Throwable v)))
                   (into {}))]
    (if (not-empty errors)
      (let [[status message] (find-error-status-and-message errors)]
        {:status (or status 520)
         :body {:message (or message "Server error")
                :problems (->> errors
                              (map (fn [[k v]]
                                     [k (or (ex-data v) (.getMessage v))]))
                              (into {}))}})
      (get result (first (keys result))))))

(defn- log [response context]
  (let [response-body (cheshire/generate-string (or (:body response) ""))]
    (logger/with-merged-config {:appenders {:json-spit {:enabled? true}
                                            :println {:enabled? false}}}
      (logger/info
       (cheshire/generate-string
        (-> (:request context)
            (update :response_time #(- (System/currentTimeMillis) %))
            (assoc :response_body response-body)
            (assoc :response_content_length (count response-body))
            (assoc :response_status (:status response))))))))

(defmethod unit/unit->unit-fn :response [_]
  (fn [{:keys [context] :as request}]
    (let [response (wrap-response request)]
      (log response context)
      response)))
