(ns volga-firebird.unit.common.http-client
  (:require [camel-snake-kebab.core :refer [->HTTP-Header-Case-String]]
            [cheshire.core :as cheshire]
            [clj-time.core :as t]
            [clojure.set :refer [rename-keys]]
            [clojure.string :as string]
            [org.httpkit.client :as http]
            [firebird-logger.core :as logger]
            [volga-firebird.unit.common.url :as url])
  (:import java.net.URI
           [javax.net.ssl SNIHostName SSLEngine SSLParameters]))

(defn- sni-configure
  [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setSSLParameters ssl-engine ssl-params)))

(defn- make-client []
  (http/make-client {:ssl-configurer sni-configure}))

(defonce ^:private client (make-client))

(defn- fix-http-method [method]
  (cond-> method
    (keyword? method) name
    true              string/lower-case
    true              keyword))

(defn- fix-headers [headers]
  (let [headers (->> headers
                    (map (fn [[header-key header-value]]
                           [(if (contains? #{"SOAPAction" "ETag" :SOAPAction :ETag} header-key)
                              (if (keyword? header-key)
                                (name header-key)
                                header-key)
                              (->HTTP-Header-Case-String header-key))
                            (str header-value)]))
                    (into {}))]
    (update headers "Content-Type" (fn [ct]
                                     (or ct "application/json")))))

(defn- fix-body [body content-type]
  (cond
    (or (empty? content-type) (nil? body))
    nil

    (and (string? content-type)
         (string/starts-with? content-type "application/json"))
    (cheshire/generate-string body)

    :otherwise
    body))

(defn- parse-body [body content-type]
  (cond
    (empty? content-type)
    body

    (and (string? body)
         (string/starts-with? content-type "application/json"))
    (cheshire/parse-string body true)

    :otherwise
    body))

(defn- parse-cookies [cookie-string]
  (when cookie-string
    (into {}
          (for [cookie (.split cookie-string ";")]
            (let [keyval (map #(.trim %) (.split cookie "=" 2))]
              [(keyword (first keyval))
               (or (second keyval) true)])))))

(defn- context->log [context]
  (-> context
      (select-keys [:project_id :request_id])
      (rename-keys {:request_id :parent_id})
      (assoc :sub_request true)))

(defn- response->log [{{{request-accept       "Accept"
                         request-content-type "Content-Type"} :headers
                        request-body                          :body
                        request-method                        :method
                        request-url                           :url
                        request-at                            :request-at
                        request-query                         :query-params} :opts
                       {response-content-type :content-type}                 :headers
                       response-body                                         :body
                       response-status                                       :status}]
  (let [request-url (url/add-query
                     (url/str->url request-url)
                     request-query)
        request-path (url/url->path request-url)
        request-host (url/url->host request-url)
        request-url-query (url/url->query request-url)]
    {:request_at              request-at
     :request_accept          request-accept
     :request_content_type    request-content-type
     :request_content_length  (count request-body)
     :request_method          request-method
     :request_scheme          (url/url->scheme request-url)
     :request_host            request-host
     :request_path            request-path
     :request_url             (str request-url)
     :request_url_query       request-url-query
     :request_body            request-body
     :response_content_type   response-content-type
     :response_content_length (count response-body)
     :response_status         response-status
     :response_body           response-body
     :response_age            0
     :response_time           (t/in-millis (t/interval request-at (t/now)))}))

(defn- log [response context]
  (logger/with-merged-config {:appenders {:json-spit {:enabled? true}
                                          :println {:enabled? false}}}
    (logger/info
     (cheshire/generate-string
      (merge (response->log response)
             (context->log context))))))

(defn request [{:keys [method url query-params body headers timeout]} context]
  (let [headers      (fix-headers headers)
        content-type (get headers "Content-Type")]
    (let [method   (fix-http-method method)
          body     (and (#{:post :put :patch} method)
                        (fix-body body content-type))
          params   {:client        client
                    :url           url
                    :method        method
                    :headers       headers
                    :query-params  query-params
                    :max-redirects 10
                    :timeout       (* (or timeout 30) 1000)
                    :request-at    (t/now)}
          params   (cond-> params
                     body (assoc :body body))
          response @(http/request params)]
      (log response context)
      (-> response
          (select-keys [:status :body :headers])
          (update :body
                  (fn [body]
                    (parse-body body
                                (get-in response [:headers :content-type]))))
          (update-in [:headers :set-cookie]
                     parse-cookies)))))
