(ns volga-firebird.unit.soap
  (:require [paos.service :as service]
            [volga-firebird.unit :as unit]
            [volga-firebird.unit.common.excel-transform :as excel-transform]
            [volga-firebird.unit.common.http-client :as http-client]
            [volga-firebird.unit.common.http-response-code :as http-response-code]
            [volga-firebird.unit.common.url :as url]))

(defmethod unit/unit->unit-fn "SOAP"
  [{{:keys [url parameters timeout valid-response-codes
            soap-envelopes soap-action soap-version]} :config}]
  (let [{:keys [query body header path]} parameters
        soap-version                     (case soap-version
                                           "1.1" :soap
                                           "1.2" :soap12)
        soap-service                     (service/->service soap-action soap-version
                                                            (:input-xml soap-envelopes)
                                                            (:output-xml soap-envelopes)
                                                            (:fault-xml soap-envelopes))
        path-transform-function          (excel-transform/->transform-function path)
        query-transform-function         (excel-transform/->transform-function query)
        body-transform-function          (excel-transform/->transform-function body)
        header-transform-function        (excel-transform/->transform-function header)]
    (fn [{:keys [context] :as request}]
      (let [query-params      (query-transform-function request)
            body              (->> request
                                  body-transform-function
                                  (service/wrap-body soap-service))
            headers           (merge (header-transform-function request)
                                     (service/soap-headers soap-service)
                                     {:content-type (service/content-type soap-service)})
            url               (->> request
                                  path-transform-function
                                  (url/interpolate url))
            params            {:method       :post
                               :url          url
                               :headers      headers
                               :query-params query-params
                               :body         body
                               :timeout      timeout}
            {status :status
             :as    response} (http-client/request params context)
            response          (update response :body
                                      #(if (= (:status response) 200)
                                         (service/parse-response soap-service %)
                                         (if (empty? (:fault-xml soap-envelopes))
                                           %
                                           (service/parse-fault soap-service %))))]
        (if (http-response-code/unexceptional? status valid-response-codes)
          response
          (throw (ex-info "HTTP Exception" response)))))))
