<img src="https://i.imgur.com/GH71uSi.png" title="zalky" align="right" width="250"/>

# Runway

[![Clojars Project](https://img.shields.io/clojars/v/io.zalky/runway?labelColor=blue&color=green&style=flat-square&logo=clojure&logoColor=fff)](https://clojars.org/io.zalky/runway)

Coding on the fly, from take-off to landing, with a tool.deps reloadable
build library.

The pieces in this library evolved over a number of years to mitigate
long-standing pain points in existing tooling, as well as fill some
gaps that appeared with the arrival of tools.deps, newer versions of
Java, and Apple Silicon chips. With this library:

1. Power up your `deps.edn` aliases:

   - [Run multiple concurrent functions](#concurrent) via merged deps
     aliases in the same runtime

   - A simple way to [load extra namespaces](#load_ns)

   - Merge in [environment variables](#env_vars) (not enabled by
     default, opt-in via peer dependency)

2. Enjoy [rock-solid live reloading](#reload) of code and lifecycle
   management of your running application:

   - The implementation is based on a fork of
     [`clojure.tools.namespace`](https://github.com/zalky/tools.namespace)
     that fixes [`TNS-6`](https://clojure.atlassian.net/browse/TNS-6),
     which greatly improves c.t.n robustness (a patch has been
     submitted)

   - The provided implementation is for a `com.stuartsierra.component`
     based system. However it can be [extended for any arbitrary build
     framework](#other_framework).

   - Choose how to reload dependent namespaces: eagerly or lazily

   - Uses the new cross-platform
     [Axle](https://github.com/zalky/axle) watcher that replaces
     [Hawk](https://github.com/wkf/hawk) (Hawk wraps a deprecated lib
     that has issues on newer Macs and newer versions of Java)

   - More robust error handling and logging during component lifecycle
     methods

   - Cleanly shutdown your application on interrupt signals

### About Reloaded Workflows

[Reloaded workflows](https://cognitect.com/blog/2013/06/04/clojure-workflow-reloaded)
are not for everyone. Fair enough!

They can be difficult to implement correctly and there are a number of
[known pitfalls](https://github.com/clojure/tools.namespace#warnings-and-potential-problems)
 with using reloaded workflows based on `clojure.tools.namespace`.

However, most of these issues are not specific to reloaded workflows,
and are things that you need to worry about in any live coding
environment. Meanwhile, the benefits that reloaded workflows bring are
significant, especially when live coding alongside large, running
applications. Having automated, enforced heuristics for how an
application behaves as your code changes allows you to a priori
eliminate a whole subset of failure points. And these failure points
are often much trickier than the known gotchas of reloaded code.

So if you're like me, and think reloaded workflows are more than worth
the effort, Runway provides a rock-solid component-based
implementation to do it.

## Quick Start

Assuming that at minimum you want an nREPL and a code watcher, put the
following in your `deps.edn` file:

```clj
{:deps    {io.zalky/runway {:mvn/version "0.2.0"}}
 :paths   ["src"]
 :aliases {:repl    {:extra-deps {nrepl/nrepl {:mvn/version "0.8.3"}}
                     :exec-fn    runway.core/exec
                     :exec-args  {runway.nrepl/server {}}}
           :watcher {:exec-fn    runway.core/exec
                     :exec-args  {runway.core/watcher {}}}}}
```

You can then start either:

```
clojure -X:repl
```

Or both concurrently

```
clojure -X:repl:watcher
```

If you use cider you might additionally want to include some nREPL
middleware in your `:repl` alias dependencies:

```clj
cider/cider-nrepl {:mvn/version "0.28.5"}   ; or whatever your cider version is
refactor-nrepl/refactor-nrepl {:mvn/version "3.5.5"}
```

Note that the above will not start a running application for you to
live code with. Read on for how to do that.

Also, make sure you have read the [simple guidelines](#reload) on how
to make your reloaded REPL experience more robust. But TLDR:

1. Keep your REPL in the default `user` namespace (see configuration
   options)

2. Always require and alias any namespaces you want to use in your
   REPL

## Running Concurrent Functions <a name="concurrent"></a>

Runway provides a means to run multiple concurrent functions via deps
aliases. This is mostly useful when these concurrent functions need
access to the same runtime, otherwise you would just run them as
separate processes. Lets say you want to start an application server,
an nrepl server, and a file watcher to reload code. Runway provides
three functions that do this for you.

Just configure a `deps.edn` that looks like the following:

```clj
{:deps    {io.zalky/runway {:mvn/version "0.2.0"}}
 :paths   ["src"]
 :aliases {:dev    {:extra-deps {nrepl/nrepl {:mvn/version "0.8.3"}}
                    :exec-fn    runway.core/exec
                    :exec-args  {runway.nrepl/server {}
                                 runway.core/watcher {}}}
           :server {:exec-fn   runway.core/exec
                    :exec-args {runway.core/go {:system my.project/app}}}}}
```

The `:exec-args` of each alias define a set of function symbol and
argument pairs. If you now run:

```
clojure -X:dev:server
```

Clojure will first merge those aliases according to the normal
semantics of -X invocation, and then pass their combined `:exec-args`
map to `runway.core/exec`. Runway will then locate the functions
declared in the combined `:exec-args` maps, load their namespaces, and
run them concurrently with their respective arguments.

Effectively the alias that gets run is:

```clj
{:extra-deps {nrepl/nrepl {:mvn/version "0.8.3"}}
 :exec-fn    runway.core/exec
 :exec-args  {runway.nrepl/server {}
              runway.core/watcher {}
              runway.core/go      {:system my.project/app}}}
```

Easy. Your function aliases are composable in any combination without
having to re-write them, and their `:exec-args` are merged in the
order in which you invoked your aliases.

Any truthy function argument like `{}` or `:arg` will be passed along
to the function, whereas any falsy value indicates that the function
should not be run. Take the alias:

```clj
{:watcher/disable {:exec-fn   runway.core/exec
                   :exec-args {runway.core/watcher false}}}
```

This is useful if you want an easy way to disable something in other
aliases. The following will be merged in order:

```
clojure -X:dev:server:watcher/disable
```

There is also the option to merge in `:exec-args` via command line
arguments. The following will merge the `:exec-args` of the two
aliases `:dev` and `:server`, along with the command line edn:

```
clojure -X:dev:server '{my.project/my-process {:my "cli_arg"}}'
```

The effective alias that gets run is:

```clj
{:extra-deps {nrepl/nrepl {:mvn/version "0.8.3"}}
 :exec-fn    runway.core/exec
 :exec-args  {runway.nrepl/server   {}
              runway.core/watcher   {}
              runway.core/go        {:system my.project/app}
              my.project/my-process {:my "cli_arg"}}}
```

To see debugging information about what namespaces, functions,
`:exec-args`, and env variables are being merged in by Runway, simply
add `:verbose "true"` to your `:exec-args`:

```
clojure -X:dev:server '{my.project/my-process {:my "cli_arg"} :verbose "true"}'

22-09-27 16:19:45 zalky INFO [runway.core:488] - No environ.core, skipping
22-09-27 16:19:45 zalky INFO [runway.core:495] - Loaded namespaces (runway.nrepl runway.core my.project)
22-09-27 16:19:46 zalky INFO [runway.core:496] - Exec fns found (runway.nrepl/server runway.core/watcher runway.core/go my.project/my-process)
22-09-27 16:19:46 zalky INFO [runway.core:497] - Exec args {runway.nrepl/server {}, runway.core/watcher {}, runway.core/go {:system my.project/app} my.project/my-process {:my "cli_arg"}}
```

Note the string quotes around `:verbose "true"`. We'll come back to
that later.

### Writing Concurrent Functions

Run functions can be defined anywhere in your code. If your function
is a long-running concurrent process that needs the main thread to
block and stay alive, then it should return a response map:

```clj
{:runway/block true}
```

Once all processes have been launched, Runway will print a boot time:

```
22-09-26 03:43:40 zalky INFO [runway.core:132] - Starting my.project/app
22-09-26 03:43:40 zalky INFO [runway.build:18] - Transitive dependency started
22-09-26 03:43:40 zalky INFO [runway.build:27] - Dependency started
22-09-26 03:43:40 zalky INFO [runway.build:36] - Dependent started
22-09-26 03:43:40 zalky INFO [runway.build:9] - Singleton started
22-09-26 03:43:40 zalky INFO [runway.core:354] - Watching system...
22-09-26 03:43:40 zalky INFO [runway.nrepl:72] - nREPL server started on port 50929 on host localhost - nrepl://localhost:50929
22-09-26 03:43:40 zalky INFO [runway.core:490] - Boot time: 2.70s
```

If you return a `:runway/ready` promise in your response map, the boot
time will not print until your promise has been delivered. For
example, you could define a function like so:

```clj
(ns my.project)

(defn my-process
  [exec-args]
  (let [ready (promise)]
    (future
      (init-process! exec-args)
      (deliver ready true)
      (run-process! exec-args))
    {:runway/block true
     :runway/ready ready}))
```

See the `runway.nrepl/server` function for a real example. You could
then configure your function in an alias:

```clj
{:my-process {:exec-fn    runway.core/exec
              :exec-args  {my.project/my-process {:my "arg"}}}}
```

And invoke it with:

```clj
clojure -X:dev:server:my-process
```

## Loading a Namespace <a name="load_ns"></a>

Qualified symbols are interpreted as a run functions by
`runway.core/exec`. _Unqualified_ symbols are interpreted as
namespaces to load. Given:

```clj
{:deps    {io.zalky/runway {:mvn/version "0.2.0"}}
 :paths   ["src"]
 :aliases {:server {:exec-fn   runway.core/exec
                    :exec-args {runway.core/go   {:system my.project/app}
                                my.project.other true}}}}
```
Then

```
clojure -X:server
```

Will run the `runway.core/go` function as well as load the
`my.project.other` namespace, presumably for side-effects.

As always, to see debugging information about what namespaces are
being loaded by Runway, use `:verbose "true"`:

```
clojure -X:server '{:verbose "true"}'
```

## Environment Variables <a name="env_vars"></a>

[`Environ`](https://github.com/weavejester/environ) is a library that
can import environment settings from a number of different sources,
including environment variables.

One thing to be aware of when using Environ is that it loads _all_
your environment variables, including potentially sensitive ones, into
memory and stores them in `environ.core/env`. For this reason, Runway
treats Environ as a peer dependency.

If you do not include Environ as a dependency in your `deps.edn` (and
it is not available on the classpath), then no variables are loaded
and all the environment features in this section will be ignored.

If you do include it, Runway provides you with a way to merge
additional env variables into `environ.core/env` using the
`:exec-args` maps in your `deps.edn` aliases:

```clj
{:deps    {io.zalky/runway {:mvn/version "0.2.0"}
           environ/environ {:mvn/version "1.2.0"}}
 :paths   ["src"]
 :aliases {:dev         {:extra-deps {nrepl/nrepl {:mvn/version "0.8.3"}}
                         :exec-fn    runway.core/exec
                         :exec-args  {runway.nrepl/server {}
                                      runway.core/watcher {}}}
           :server      {:exec-fn   runway.core/exec
                         :exec-args {runway.core/go {:system my.project/app}
                                     :my-env-var-1  "val1"}}
           :server/opts {:exec-args {:my-env-var-2 "val2"}}}}
```

Any keywords in your `:exec-args` maps are interpreted as env
variables that should be merged into your `environ.core/env` map. If
you now invoke your programs with:

```
clojure -X:repl:server:server/opts
```

You should have access to both `:my-env-var-1` and `:my-env-var-2` in
`environ.core/env`:

```clj
(require '[environ.core :as env])

(select-keys env/env [:my-env-var-1 :my-env-var-2])
=>
{:my-env-var-1 "val1", :my-env-var-2 "val2"}
```

Your alias variables override any values that may have been exported
directly in your environment. For example, if you invoke:

```
clojure -X:repl:server
```

Then the value of `:my-env-var-1` will always be `"val1"`, no matter
what is `export`ed by your environment. Meanwhile the value of
`:my-env-var-2` would depend whether or not you had `export`ed it as
an env VARIABLE:

```
export MY_ENV_VAR_2=env_val
```

The above mechanism provides you with an easy way to configure your
application across development and production environments.

To see debugging information about what env variables are being merged
in by Runway, use `:verbose "true"` (this will _not_ print the full
set of variables loaded by `environ.core`):

```
clojure -X:repl:server:server/opts '{:verbose "true"}'

22-09-27 17:15:30 zalky INFO [runway.core:494] - Merged env {:my-env-var-1 "val1", :my-env-var-2 "val2", :verbose "true"}
22-09-27 17:15:31 zalky INFO [runway.core:495] - Loaded namespaces (runway.nrepl runway.core)
22-09-27 17:15:31 zalky INFO [runway.core:496] - Exec fns found (runway.nrepl/server runway.core/watcher runway.core/go)
22-09-27 17:15:31 zalky INFO [runway.core:497] - Exec args {runway.nrepl/server {}, runway.core/watcher {}, runway.core/go {:system my.project/app}}
22-09-27 17:15:31 zalky INFO [runway.core:377] - Watching system...
22-09-27 17:15:31 zalky INFO [runway.core:137] - Starting my.project/app
```

Of course, you can merge in variables via your cli `:exec-args`:

```
clojure -X:repl:server:server/opts '{:verbose "true" :my-cli-var "other"}'
22-09-27 17:24:35 zalky INFO [runway.core:494] - Merged env {:my-env-var-1 "val1", :my-env-var-2 "val2", :verbose "true", :my-cli-var "other"}
```

### Env Variable Values Must Be Strings

`:exec-args` are necessarily parsed as edn, and therefore so are the
values in your `:exec-args` maps. However, env variables that you
export in your environment are not:

```
export MY_CLI_VAR=false
```

The above will be loaded as the string `"false"`, which is actually a
truthy value in your running application. Therefore to preserve the
semantics of environment variables, Runway will throw an error if you
try to pass a non-string value to an `:exec-args` env var:

```
clojure -X:repl:server:server/opts '{:my-cli-var false}'
Exception in thread "main" java.lang.Exception: :exec-args environment key :my-cli-var error: value false must be a string
```

This is why we have been passing `:verbose "true"` throughout these
examples, and not `:verbose true`.

## Reloaded Workflow <a name="reload"></a>

Reloaded workflows with Runway can be extremely robust. Three simple
guidelines are all you need to help mitigate pitfalls:

1. Stay in the `user` namespace. The `user` namespace is special for
   two reasons:

   First, it is usually not backed by a file. Namespaces backed by
   files are in constant flux due to c.t.n., and if you move your REPL
   into one like so:

   ```clj
   (in-ns 'ns.backed.by.file)
   ```

   You will find it hard to persist your REPL vars across
   reloads. Whereas, by staying in a namespace that is not backed by a
   file, and therefore not reloaded, your REPL vars will be preserved.
   
   Second, Runway keeps your aliases and refers in `user` consistent
   with the changing namespace graph. If you move your REPL into
   another namespace, your aliases and refers may end up resolving to
   stale dependencies.

   If for whatever reason your default REPL namespace is not `user`,
   or if your `user` namespace happens to be backed by a file, you can
   configure the Runway watcher with another REPL namespace via the
   `:repl-ns` option described below. Whatever this namesapce is, stay
   in it.

2. Always require and alias any namespace you want to use in your
   REPL.

3. You must change the contents of a file to trigger a reload. Simply
   saving a file without modifying it will have no effect.

Next, some caveats about live-coding that are not specific to reloaded
workflows, but are nevertheless critical to be aware of:

1. Be careful what you `def` in your REPL namespace. _Especially
   objects that implement protocols or interfaces, like system
   components!_ Anything you `def` into your REPL becomes a snapshot
   of your code at a point in time, and can easily get out of sync as
   your other namespaces change. And while Runway updates your REPL
   aliases and refers, it cannot account for stale REPL state that you
   may have captured through `def`s or closures.

   If you are not sure whether the thing you are `def`ing into your
   REPL can become stale, then consider using a `defn` instead.
   
2. `defmethod`s will be reloaded, but cannot be outright removed. If
   you really want a stale `defmethod` gone, the fool-proof approach
   is to trigger a reload on the file that contains the `defmulti`
   MultiFn. `remove-method` can also work, but there are some edge
   cases if you also use `prefer-method`.

3. Runway's code reloading is robust enough that you should be able to
   switch branches that are any distance apart, as long as the
   classpath doesn't change. However, any changes with implications on
   the classpath are likely to cause problems.

### Configuration

Typically you would start at least a development server, a code
watcher, and an nREPL server. Something like this in your `deps.edn`
aliases would work:

```clj
{:dev    {:extra-deps {nrepl/nrepl       {:mvn/version "0.8.3"}
                       cider/cider-nrepl {:mvn/version "0.28.5"}             ; optional
                       refactor-nrepl/refactor-nrepl {:mvn/version "3.5.5"}} ; optional
          :exec-fn    runway.core/exec
          :exec-args  {runway.nrepl/server {}
                       runway.core/watcher {}}}
 :server {:exec-fn   runway.core/exec
          :exec-args {runway.core/go {:system my.project/app}}}}
```

You could start any combination of the above aliases from the command
line:

```
clojure -X:dev:server
```

#### `runway.core/watcher`

Watches your Clojure files, reloads their corresponding namespaces
when they change, and then if necessary, restarts the running
application. It is configurable with the following options:

1. **`:watch-fx`**: accepts a sequence of `runway.core/do-fx`
   multimethod dispatch values. See `runway.core/watcher-config` for
   the default list. You can extend watcher functionality via the
   `runway.core/do-fx` multimethod. Each method behaves like an
   interceptor: it accepts a c.t.n. tracker map and returns and
   updated one, potentially modifying the remaining fx chain. Just
   take care: the order of fx methods is not necessarily commutative.

2. **`:repl-ns`**: if your starting REPL namespace is different than
   the default `user`, you can tell Runway where to update REPL
   aliases and refers. Note that Runway does nothing to ensure that
   this namespace exists, and that it is the current namespace. That
   part is up to you. Make sure you've done this before the first
   reload or you will get a watcher error.

3. **`:lazy-dependents`**: whether to load dependent namespaces
   eagerly or lazily. See [the section on reloading
   heuristics](#reload_heuristics) for more details.

4. **`:restart-fn`**: A symbol to a predicate function that when
   given a c.t.n. tracker and a namespace symbol, returns a boolean
   whether or not the system requires a restart due to the reloading
   of that namespace. This allows you to override why and when the
   running application is restarted.

4. **`:restart-paths`**: A list of paths that should trigger an
   application restart on change. Note that the paths can be either
   files or directories, with directories also being matched on
   changed contents. This is useful if your reloaded workflow depends
   on static resources that are not Clojure code, but may affect the
   running application. For example, let's say your application is
   configured via edn:
   
   ```
   {:exec-fn   runway.core/exec
    :exec-args {runway.core/watcher {:restart-paths ["config/edn/"]}}}
   ```

#### `runway.core/go`

Starts the application once on boot (after boot, you can start or stop
the application manually or via the watcher). It has the following
options:

1. **`:system`**: A symbol that refers to a function that when called
   with no arguments, returns a `com.stuartsierra.component/SystemMap`
   ([see here on how to extend for other build
   frameworks](#other_framework)). In the example above it would be a
   function called `my.project/app`.

1. **`:shutdown-signals`**: A sequence of strings representing POSIX
   interrupt signals (default is `["SIGINT" "SIGTERM" "SIGHUP"]`). On
   receiving such a signal Runway will first attempt to shutdown the
   application before re-raising.

#### `runway.nrepl/server`

Launches an nREPL server for you to connect to, and has the following
options:

1. **`:port`**: nREPL port where to listen to for connections. For
   example you could configure a specific port directly from the
   command line like so:

   ```clj
   clojure -X:dev:server '{runway.nrepl/server {:port 50000}}'
   ```

2. **`:middleware`**: A list of nREPL middleware symbols. Each symbol
   can either directly reference a middleware function, or point to a
   list of more symbols. For example:
   
   ```clj
   {runway.nrepl/server {:middleware [my.project.nrepl/my-middleware-list]}}
   ```

   `cider.nrepl` and `refactor-nrepl.middleware` is special: if no
   `:middleware` option is provided, but either is on the classpath,
   then their default middleware sets are loaded automatically. So all
   you have to do is include `cider/cider-nrepl` or
   `refactor-nrepl/refactor-nrepl` in your deps, and they should
   work. See `runway.nrepl/default-middleware` for the full list of
   default middleware.

   *A note to Cider users*: the `cider.nrepl/wrap-out` middleware
   re-directs all stdout to the Cider REPL, spamming the REPL with log
   ouput from the running system. With any kind of complex application
   this makes the REPL unusable, so it has been removed from the
   default middleware list. Of course you can always add it back with
   the `:middleware` option as described above.

### Source Directories

Any directories on your classpath will be searched for Clojure files
by the Runway watcher. Use the `:paths` in your `deps.edn` aliases to
determine what is watched.

### Reload Heuristics <a name="reload_heuristics"></a>

While live coding you can usually rely on runway to at minimum reload
your changed namespaces in dependency order, and additionally reload
the transitive dependents of those changed namespaces. However, the
behaviour of transitive dependents loading can be configured to be
either eager (the default), or lazy.

To understand what this means, we first have to understand the
difference between the namespace dependency graph that
`clojure.tools.namespace` computes from the contents of your source
files, and the actual dependency graph between your namespace objects
in memory.

When you first load your application with the `runway.core/go` task,
only the subset of the code that is required to construct your
application is loaded into memory. Likewise, if you start just a REPL,
only those namespaces that you `require` in your REPL will be
loaded. But once you start a `runway.core/watcher` task, and by
extension a `clojure.tools.namespace` tracker, they will automatically
reload _all_ dependents of a changed namespace that are found
_anywhere in your source directories_, including dependents that have
until that point not been required either by the running application
or your REPL.

For example, let's say you've defined your running application in
namespace `b`, which has a single dependency `a`.

```
  a
 /
b     ; b defines your running app and requires a
```

There's another source file, `c`, that is not required for your
running application, but also requires `a`.

```
  a
 / \
b   c
```

When you start your app with `runway.core/go`, and a
`runway.core/watcher` task, initially only `a` and `b` are loaded into
memory.

However as soon as you modify `a` on disk, both dependents `b` and `c`
are eagerly reloaded by the watcher, even though `c` was not initially
required by your running application. The heuristics are the same if
you start a REPL and a watcher, evaluate `(require 'b)`, and then
modify `a` on disk: `c` will be eagerly reloaded.

The idea is that for any change to source, you want to realize _all_
dependent effects right away. You do not want to wait for changes to
accumulate, only to later find out that conflicts or errors have been
introduced and have become more difficult to resolve. So the eager
realization of dependent source changes is almost always desirable.

However, it may be that loading namespace `c` produces some
side-effects that you'd rather not have happen (side-effects in your
namespaces are best avoided, but such is life). If you really need,
you can configure the Runway watcher to load dependents lazily via the
`:lazy-dependents` option.

With `:lazy-dependents true`, the watcher would not automatically
reload `c` unless it has already been loaded once by some other means,
either as part of the application boot, or explicitly from the REPL
with `(require 'c)`. From that point on, any changes to `a`, would
automatically cause `c` to reload.

### Component Lifecycles

Just before reloading code, Runway also checks whether any of the
namespaces it is about to update depends directly on
`com.stuartsierra.component`. These namespaces usually define system
components that implement the `com.stuartsierra.component/Lifecycle`
protocol and participate in the running application. Before reloading
any such namespaces, Runway will first stop the running
application. Then if all namespaces are successfully reloaded, Runway
will start the running application again. This process ensures that
all the stateful objects that were loaded as part of your running
application are consistent with your updated code.

At any point, you can manually stop, start or restart your running
application from the REPL with:

```clj
(require '[runway.core :as run])

(run/stop)
(run/start)
(run/restart)
```

If you need acces to your running application during live coding, it
resides in the `runway.core/system` var. But remember, never access
this var or its contents outside of your REPL work: doing so by-passes
the component dependency graph and is considered a component
anti-pattern.

### On Errors

Runway handles errors in each phase of a reload in different ways:

1. **Component failed `start` lifecycle**: Runway will abort the
   start, and then attempt to recover by stopping the components of
   the system it had already started up to that point.

2. **Component failed `stop` lifecycle**: Runway will abort stopping
   the problem component, but attempt to recover by stopping all other
   components that are not transitive dependents of the problem
   component (these should already have been stopped).

3. **Namespace reloading failed**: Runway will not restart the running
   application until all errors have been resolved.

### Logging

Runway implements logging via the excellent
[`com.taoensso/timbre`](https://github.com/ptaoussanis/timbre) library
for maximum extensibility.

Specifically, Lifecycle exceptions are logged out with full component
and system data. However, because components in complex systems can be
quite large, you may want to truncate such output to avoid spamming
terminals and log sinks. You can easily do so using custom timbre
appenders.

## System Definition

You can make your `com.stuartsierra.component/SystemMap` any way you
like, but Runway also provides some facilities that make it a bit
easier:

```clj
(ns my.project
  (:require [runway.core :as run]))

(def base-components
  {:dependency [->Dependency arg1 arg2]
   :dependent  [->Dependent]})

(def base-dependencies
  {:dependent [:dependency]})

(defn base-system
  "Symbol that gets passed to runway.core/go"
  []
  (run/assemble-system base-components base-dependencies))
```

Here `->Dependency` is any constructor function that when applied to
its arguments `arg1 arg2`, returns a component that implements the
`com.stuartsierra.component/Lifecycle` protocol.

Defined as such, you can easily re-combine systems in arbitrary ways:

```clj
(def fullstack-components
  (->> {:new-dependency [->NewDependency]}
       (merge base-components)))

;; => {:dependency     [->Dependency arg1 arg2]
;;     :dependent      [->Dependent]
;;     :new-dependency [->NewDependency]}

(def fullstack-dependencies
  (->> {:dependent [:new-dependency]}
       (run/merge-deps base-dependencies)))

;; => {:dependent [:dependency :new-dependency]}

(defn fullstack-system
  []
  (run/assemble-system fullstack-components fullstack-dependencies))
```

### System Component Library

You can find a number of useful, ready made components in the
[System](https://github.com/danielsz/system) library.

## Main Invocation

You may want to configure your project with a `-main` function. Runway
provides CLI arg parsing for the `runway.core/go` method via
`runway.core/cli-args` (implemented using `clojure.tools.cli`). For
example you could write something like:

```clj
(ns my.project
  (:require [runway.core :as run]))

(defn -main
  [& args]
  (let [{options :options
         summary :summary
         :as     parsed} (run/cli-args args)]
    (if (:help options)
      (println summary)
      (do (run/go options)
          @(promise)))))
```

Here, `args` are CLI args meant for `runway.core/go` (at mininum
`--system my.project/app`), and not args to your running
application. To configure your actual application, prefer environment
variables, edn, or a configuration framework like Zookeeper.

## Other Build Frameworks <a name="other_framework"></a>

The default build implementation provided by Runway is for
`com.stuartsierra.component/SystemMap`. If you use some other build
framework to start and stop your running application, simply wrap your
system in a record that implements
`com.stuartsierra.component/Lifecycle`, and
`runway.core/IRecover`. You probably also want to set a custom
`:restart-fn` watcher predicate. With the default `:restart-fn`
predicate, Runway restarts your app whenever a direct dependent of
`com.stuartsierra.component` changes, which is probably not what you
want for a non-Component build framework.

Something like this should work:

```clj
(ns my.project
  (:require [com.stuartsierra.component :as component]
            [runway.core :as run]))

(defrecord ComponentWrapper [impl-system]
  component/Lifecycle
  (start [_]
    (try
      (->ComponentWrapper (impl-start-fn impl-system))
      (catch Throwable e
        (let [id  (get-failed-id e)
              sys (get-failed-system e)]
          ;; Re-throw as component error
          (throw
           (ex-info (get-error-msg e)
                    {:system-key id
                     :system     (->ComponentWrapper sys)
                     :function   #'impl-start-fn}
                    e))))))

  (stop [_]
    (try
      (->ComponentWrapper (impl-stop-fn impl-system))
      (catch Throwable e
        (let [id  (get-failed-id e)
              sys (get-failed-system e)]
          ;; Re-throw as component error
          (throw
           (ex-info (get-error-msg e)
                    {:system-key id
                     :system     (->ComponentWrapper sys)
                     :function   #'impl-stop-fn}
                    e))))))

  run/Recover
  (recoverable-system [_ failed-id]
    (-> impl-system
        (impl-recoverable-subsystem)
        (->ComponentWrapper))))

(defn wrapped-app
  "Passed to runway.core/go"
  []
  (->ComponentWrapper (impl-system-constructor)))
```

There are five important things to note:

1. On error you need to re-throw a component error. A component error
   is of type `clojure.lang.ExceptionInfo` and contains at minimum:

   - `:system-key`: This is the failed component id
   - `:system`: This is the failed implementation system, wrapped in a
   `ComponentWrapper`
   - `:function`: This is the lifecycle that failed, only used for logging
   
   Both the `:system-key` and the wrapped `:system` are passed to your
   `recoverable-system` implementation, which needs to return a
   wrapped subsystem that Runway will attempt to stop to recover from
   the error. This subsystem should include everything that still
   needs to be stopped _excluding_ the failed component.

   If your `impl-start-fn` and `impl-stop-fn` do not throw errors, but
   instead return errors as data, simply parse the data and then throw
   a component exception.

2. On success there's not much to do. Just return the started or
   stopped implementation system, making sure it is wrapped.

3. Wherever we delegate back to Runway, our implementation system must
   always be wrapped in a `ComponentWrapper`. This includes the return
   values of each protocol method, as well as in the re-thrown
   component exception. It also includes the recoverable subsystem and
   the constructor function passed to `runway.core/go`.

4. If there's no need to recover your system on lifecycle errors, or
   maybe `impl-start-fn` and `impl-stop-fn` handle recovery
   directly, you can always choose not to re-throw errors in the
   `start` and `stop` `Lifecycle` methods. In this case
   `recoverable-system` is never called and `ComponentWrapper` becomes
   trivial.

5. Make sure you do not accidentally double wrap the component.

See `runway.wrapped` for a working stub that implements this full
pattern where the other "framework" is just a simple map.

## License

Runway is distributed under the terms of the Apache License 2.0.
