(ns swim.feed-index
  "Functions for creating an index of all messages in the feed."
  (:require [swim.utils :refer [remove-last removev]]
            [swim.parse :refer [parse-subject]]))

(defn order-key
  "Put id in :order, sha in :sent."
  [id]
  (if (string? id) :sent :order))

(defn append-to-order
  "Add the given message-id to the end of the order for the given feed or thread."
  [feed id]
  (update-in feed [(order-key id)] (fnil conj []) id))

(defn remove-from-order
  "Remove the given message-id from the order for the given feed or thread by starting at the end
  and searching backwards."
  [feed id]
  (update-in feed [(order-key id)] remove-last #{id}))

(defn get-message
  "Get a message by id from inside of a thread."
  [thread id]
  (get-in thread [:messages id]))

(defn get-thread
  "Get a thread by thread-id from a feed."
  [feed thread-id]
  (get-in feed [:threads thread-id]))

(defn remove-sent-message
  "Remove the message with matching sha if it exists."
  [thread sha]
  (if (get-message thread sha)
    (-> thread
        (remove-from-order sha)
        (update-in [:messages] dissoc sha))
    thread))

(defn append-message
  "Append a message to the end of a thread if it doesn't already exist. If it already exists, this
  is just an edit, so replace it."
  [thread message]
  (let [{:keys [id sha]} message]
    (-> thread
        (remove-sent-message sha)
        (cond-> (not (get-message thread id))
                (append-to-order id))
        (assoc-in [:messages id] message))))

(defn append-thread-message
  [feed message]
  (let [thread-id (:thread-id message)]
    (-> feed
        (assoc-in [:threads thread-id :id] thread-id)
        (update-in [:threads thread-id] append-message message))))

(defn replace-sha-thread-id
  "Update a thread originally sent from here by replacing sha with the server-assigned thread-id."
  [feed sha thread-id]
  (if-let [sha-thread (get-thread feed sha)]
    (-> feed
        (update-in [:threads] dissoc sha)
        (assoc-in [:threads thread-id] (assoc sha-thread :id thread-id))
        (remove-from-order sha))
    feed))

(defn move-thread-to-top
  "Remove thread from order if it is already there, and add it at the top."
  [feed thread-id]
  (-> feed
      (cond-> (get-thread feed thread-id)
              (remove-from-order thread-id))
      (append-to-order thread-id)))

(defn add-thread-subject
  "Parse thread subject from body and add it if there is one."
  [feed message]
  (let [{:keys [id thread-id sha body subject]} message]
    (if-let [subject (or subject (when (= thread-id sha)
                                   (parse-subject body)))]
      (assoc-in feed [:threads thread-id :subject] subject)      
      feed)))

(defn add-participants
  "Add new thread participants from this message."
  [feed message]
  (let [{:keys [thread-id author-id body]} message
        participants (:participant-ids (get-thread feed thread-id) [])
        existing (set participants)]
    (if-let [new (seq (->> (conj (keep :user body) author-id)
                           (remove existing)
                           (distinct)))]
      (assoc-in feed [:threads thread-id :participant-ids] (into participants new))
      feed)))

(defn add-new-message
  "Add a new message by appending it to the appropriate thread and moving that thread to the top of
  the feed order."
  [feed message]
  (let [{:keys [sha thread-id]} message]
    (-> feed
        (replace-sha-thread-id sha thread-id)
        (move-thread-to-top thread-id)
        (append-thread-message message)
        (add-thread-subject message)
        (add-participants message))))

(defn update-thread-attrs
  "Update participants or subject in a specific thread."
  [feed thread-id attrs]
  (update-in feed [:threads thread-id] into
             (select-keys attrs [:participant-ids :subject])))

(def empty-feed
  {:threads {}
   :order []
   :sent []})

(defn add-feed-order
  "Calculate the feed order based on the latest message in each thread. This assumes that there are
  no sent messages with sha ids yet."
  [feed]
  (assoc feed
    :order (vec (keys (sort-by (comp peek :order val)
                               (:threads feed))))))

(defn init-feed
  "Initialize the feed index based on an ordered list of server messages."
  [messages]
  (-> (reduce append-thread-message empty-feed messages)
      (add-feed-order)))

(defn backfill-thread
  "Backfill thread given a backfill map of old messages containing :messages and :order."
  [thread backfill]
  (assoc thread
    :messages (merge (:messages backfill)
                     (:messages thread))
    :order (into (removev (:messages thread)
                          (:order backfill))
                 (:order thread))))

(defn backfill-feed
  "Backfill feed given a backfill map of old threads containing :threads and :order."
  [feed backfill]
  (assoc feed
    :order (into (removev (:threads feed)
                          (:order backfill))
                 (:order feed))
    :threads (merge-with backfill-thread
                         (:threads feed)
                         (:threads backfill))))
