(ns swim.parse.message
  (:refer-clojure :exclude [time])
  (:require [clojure.string :refer [lower-case]]
            [swim.parse.core
             :refer [<< >> all digit either ignore int* is-not letter many maybe one prefix
                     repeated scan str* symbol-char token transform verify whitespace-char
                     word-char with-consumed]]))

(defn identifier []
  (str*
   (all (str* (many (digit)))
        (letter)
        (str* (many
               (either (letter)
                       (digit)
                       (token #{\_ \-})))))))

(defn tag []
  (-> (>> (one \#) (identifier))
      (transform (fn [tag]
                   {:type :tag
                    :string tag}))))

(defn email []
  (-> (str*
       (all (str*
             (many 1 (either (symbol-char)
                             (token #{\+ \.}))))
            (one \@)
            (str* (many 1 (symbol-char)))
            (str*
             (many 1 (str*
                      (prefix (one \.)
                              (many 1 (word-char))))))))
      (transform (fn [email]
                   {:type :email
                    :identifier (lower-case email)}))))

(def min-phone 10)

(defn phone []
  (-> (str*
       (all (maybe (one \+))
            (str*
             (many (either (digit)
                           (ignore (is-not (letter))))))))
      (verify (fn [number]
                (<= min-phone (count number))))
      (transform (fn [number]
                   {:type :phone
                    :identifier (case (first number)
                                  \+ number
                                  \1 (str "+" number)
                                  (if (= min-phone (count number))
                                    (str "+1" number)
                                    (str "+" number)))}))))

(defn address []
  (either (email) (phone)))

(defn- mmap [m]
  (into {}
        (for [[ks v] m, k ks]
          [k v])))

(def str->units
  (mmap
   {["s" "sec" "secs" "second" "seconds"] :seconds
    ["m" "min" "mins" "minute" "minutes"] :minutes
    ["h" "hour" "hours"] :hours
    ["d" "day" "days"] :days
    ["w" "week" "weeks"] :weeks
    ["month" "months"] :months
    ["y" "yr" "yrs" "year" "years"] :years}))

(def str->period
  (mmap
   {["a" "am"] :am
    ["p" "pm"] :pm}))

(defn interval []
  (-> (all (int* (many 1 (digit)))
           (str* (many 1 (letter))))
      (transform (fn [[num units]]
                   (when-let [units (str->units (lower-case units))]
                     {:type :interval
                      :num num
                      :units units})))
      (verify (complement nil?))))

(defn time []
  (-> (all (int* (many 1 2 (digit)))
           (maybe (>> (one \:)
                      (int* (repeated 2 (digit)))))
           (maybe (str* (many 1 2 (letter)))))
      (transform (fn [[hour minute string]]
                   (let [period (and string (str->period (lower-case string)))]
                     (when (or period (nil? string))
                       {:type :time
                        :hour hour
                        :minute minute
                        :period period}))))
      (verify (complement nil?))))

(defn date []
  (-> (all (int* (repeated 4 (digit)))
           (many 0 2 (>> (token #{\- \/})
                         (int* (many 1 2 (digit))))))
      (transform (fn [[year [month day]]]
                   {:type :date
                    :year year
                    :month month
                    :day day}))))

(defn short-date []
  (-> (all (int* (many 1 2 (digit)))
           (>> (token #{\- \/})
               (int* (many 1 2 (digit)))))
      (transform (fn [[month day]]
                   {:type :date
                    :month month
                    :day day}))))

(defn date-time []
  (-> (all (either (date)
                   (short-date))
           (>> (token #{\T \-})
               (time)))
      (transform (fn [[date time]]
                   (merge time date)))))

(defn command []
  (-> (>> (one \^) (either (interval)
                           (date-time)
                           (time)))
      (with-consumed)
      (transform (fn [[delay consumed]]
                   {:type :command
                    :delay delay
                    :raw (apply str consumed)}))))

(def sentence-terminator (token #{\? \! \.}))

(defn sentence [& {:keys [stop-at-newline require-terminator]}]
  (let [terminator (cond-> (<< sentence-terminator (whitespace-char))
                     stop-at-newline (either (one \newline)))]
    (-> (scan terminator)
        (cond-> require-terminator
          (verify (fn [[sentence end]] end)))
        (transform (fn [[sentence end]]
                     (apply str (cond-> sentence
                                        (not= \newline end) (conj end))))))))

;;;;;;;;;;;; This file autogenerated from src/swim/parse/message.cljx
