(ns chord.format
  (:require       [cheshire.core :as json]
            
                  [clojure.core.async :as a]
                                          
            
                  [clojure.java.io :as io]

                  [clojure.tools.reader.edn :as edn]
                                        
            
            [clojure.walk :refer [keywordize-keys]]
            [cognitect.transit :as transit])

        (:import [java.io ByteArrayOutputStream ByteArrayInputStream]))

(defprotocol ChordFormatter
  (freeze [_ obj])
  (thaw [_ s]))

(defmulti formatter* :format)

(defmethod formatter* :edn [_]
  (reify ChordFormatter
    (freeze [_ obj]
      (pr-str obj))

    (thaw [_ s]
      (edn/read-string s))))

(defmethod formatter* :json [_]
  (reify ChordFormatter
    (freeze [_ obj]
            (json/encode obj)
                                              )

    (thaw [this s]
            (json/decode s)
                                        )))

(defmethod formatter* :json-kw [opts]
  (let [json-formatter (formatter* (assoc opts :format :json))]
    (reify ChordFormatter
      (freeze [_ obj]
        (freeze json-formatter obj))
      
      (thaw [_ s]
        (keywordize-keys (thaw json-formatter s))))))

(defmethod formatter* :transit-json [_]
  (reify ChordFormatter
    (freeze [_ obj]
           
      (let [baos (ByteArrayOutputStream.)]
        (transit/write (transit/writer baos :json) obj)
        (.toString baos))

            
                                                )

    (thaw [_ s]
           
      (let [bais (ByteArrayInputStream. (.getBytes s))]
        (transit/read (transit/reader bais :json)))

            
                                             )))

(defmethod formatter* :str [_]
  (reify ChordFormatter
    (freeze [_ obj]
      obj)

    (thaw [_ s]
      s)))

(defn formatter [opts]
  (formatter* (if (keyword? opts)
                {:format opts}
                opts)))

(defn wrap-format [{:keys [read-ch write-ch]} {:keys [format] :as opts}]
  (let [fmtr (formatter (if format
                          opts
                          {:format :edn}))]

    ;; TODO need to replace a/map< etc with transducers when 1.7.0 is
    ;; released
    
    {:read-ch (a/map< (fn [{:keys [message]}]
                        (try
                          {:message (thaw fmtr message)}
                          (catch       Exception                 e
                                 {:error :invalid-format
                                  :cause e
                                  :invalid-msg message})))
                      read-ch)

     :write-ch (a/map> #(freeze fmtr %) write-ch)}))

;;;;;;;;;;;; This file autogenerated from src/chord/format.cljx
