(ns jlk.batik.core
  (use [seesaw.core :only [invoke-later frame pack! show!]])
  (import [org.apache.batik.swing JSVGCanvas]
          [org.apache.batik.swing.svg JSVGComponent GVTTreeBuilderAdapter]
          [org.apache.batik.dom.svg SVGDOMImplementation]))

(defn svg-format-keyword
  [x]
  (if (instance? clojure.lang.Named x)
    (name x)
    (str x)))

(defn dom
  ""
  []
  (SVGDOMImplementation/getDOMImplementation))

(defn uri
  []
  SVGDOMImplementation/SVG_NAMESPACE_URI)

(defn set-attribute
  "n,v implementing named converted using name, otherwise conversion by str"
  [e n v]
  (.setAttributeNS e nil (svg-format-keyword n) (svg-format-keyword v)))

(defn root
  ""
  [so & attrs]
  (let [r (.getDocumentElement (:document so))]
    (doseq [[k v] (partition 2 attrs)]
      (set-attribute r k v))
    r))

(defn document
  ""
  []
  (.createDocument (dom) (uri) "svg" nil))

(defrecord SO [canvas document])
(defn so
  []
  (let [canvas (JSVGCanvas.)
        document (document)]
    ;; ensure that an UpdateManager is available by calling the following...
    (.setDocumentState canvas JSVGCanvas/ALWAYS_DYNAMIC)
    (set-attribute (.getDocumentElement document) :width 300)
    (set-attribute (.getDocumentElement document) :height 300)
    (.setSVGDocument canvas document)
    (SO. canvas document)))

(def aso (so))

(defn paint
  [so]
  (.setSVGDocument (:canvas so) (:document so)))

(defmacro doupdate
  "this will nullpointer if updatemanager has not been created..."
  [so & body]
  `(-> ~so :canvas .getUpdateManager .getUpdateRunnableQueue (.invokeLater (fn [] ~@body))))

(defn element
  ""
  [e n & attrs]
  (let [en (.createElementNS (.getDocument e) (uri) (name n))]
    (doseq [[k v] (partition 2 attrs)]
      (set-attribute en k v))
    (.appendChild e en)))

(defn svg-format-points
  "take a sequence of number pairs return as a svg friendly string"
  [points]
  (reduce (fn [s [a b]] (format "%s %s,%s" s a b)) "" (partition 2 points)))

;; use these helper methods to help document the SVG interface

(defn -line
  [e {:keys [x1 y1 x2 y2] :as opts}]
  (apply element e :line (flatten (seq opts))))

(defn -poly
  [e {:keys [points] :as opts}]
  (apply element e :polyline (flatten (seq (assoc opts :points (svg-format-points points))))))

(defn -rect
  "(rectangle (root so)), (paint so)"
  [e {:keys [x y width height rx ry] :as opts}]
  (apply element e :rect (flatten (seq opts))))

(defn -circ
  [e {:keys [cx cy r] :as opts}]
  (apply element e :circle (flatten (seq opts))))

(defn -main
  ([]
     (-main aso))
  ([so]
     (invoke-later
       (-> (frame :title "hello"
                  :content (:canvas so))
           pack!
           show!))))

(defn test-core
  []
  (-main)
  (doupdate aso
    (-line (root aso) {:x1 0 :y1 0 :x2 200 :y2 100 :stroke :red :stroke-width 10})
    (-rect (root aso) {:x 200 :y 50 :width 10 :height 10 :color :black})
    (-poly (root aso) {:points [0,0 200,200 50,300] :stroke :blue :fill :none})
    ;; move this
    ;;(poly (root aso) :points [[0,50] [200,250] [50,350]] :stroke :pink :fill :none)
    (-circ (root aso) {:cx 150 :cy 150 :r 20})))

;;
;; provide an extensible cad like user interface
;; this can be extended to apache vectors etc.
;;

(defmulti line (fn [_ A B & _] [(class A) (class B)]))
(defmethod line [clojure.lang.PersistentVector clojure.lang.PersistentVector]
  [e A B & opts]
  (-line e (assoc (apply hash-map opts) :x1 (first A) :y1 (second A) :x2 (first B) :y2 (second B))))

(defmulti poly (fn [_ points & _] (class points)))
(defmethod poly clojure.lang.PersistentVector
  [e points & opts]
  (-poly e (assoc (apply hash-map opts) :points (flatten points))))

(defmulti rectp (fn [_ A B & _] [(class A) (class B)]))
(defmethod rectp [clojure.lang.PersistentVector clojure.lang.PersistentVector]
  [e A B & opts]
  (-rect e (assoc (apply hash-map opts)
             :x (first A)
             :y (second A)
             :width (Math/abs (- (first B) (first A)))
             :height (Math/abs (- (second B) (second A))))))

(defmulti rectd (fn [_ A B & _] [(class A) (class B)]))
(defmethod rectd [clojure.lang.PersistentVector clojure.lang.PersistentVector]
  [e A B & opts]
  (-rect e (assoc (apply hash-map opts)
             :x (first A)
             :y (second A)
             :width (first B)
             :height (second B))))

(defmulti circr (fn [_ A _ & opts] (class A)))
(defmethod circr clojure.lang.PersistentVector
  [e A r & opts]
  (-circ e (assoc (apply hash-map opts)
             :cx (first A)
             :cy (second A)
             :r r)))

(defmulti circd (fn [_ A _ & opts] (class A)))
(defmethod circd clojure.lang.PersistentVector
  [e A d & opts]
  (-circ e (assoc (apply hash-map opts)
             :cx (first A)
             :cy (second A)
             :r (/ d 2))))

(defn test-user
  []
  (-main)
  (doupdate aso
    (line (root aso) [100 100] [200 250] :stroke :red :stroke-width 3)
    (poly (root aso) [[100 100] [200 220] [300 300]] :stroke :pink :stroke-width 2 :fill :none)
    (rectp (root aso) [100 100] [200 200] :stroke :red :stroke-width 3 :fill :none)
    (rectd (root aso) [100 100] [50 50] :stroke :black :fill :none)
    (circr (root aso) [200 200] 20 :stroke :blue :fill :none)
    (circd (root aso) [200 200] 20 :stroke :yellow :fill :none)))
