(ns jlk.data.office.excel
  (:import [java.io FileInputStream FileOutputStream]
           [org.apache.poi.ss.usermodel WorkbookFactory Cell CellStyle]
           [org.apache.poi.hssf.usermodel HSSFWorkbook]
           [org.apache.poi.ss.util WorkbookUtil]))

;;
;; apache poi is a library for accessing microsoft office files
;; http://poi.apache.org/
;; 

;;
;; NOTES:
;; - make sure the following additional files are in the classpath
;; - ooxml and ooxml-schemas
;; - xmlbeans (xbean.jar)
;; - log4j
;; - dom4j
;;

(defn workbook
  "return a handle to a workbook.  if :file is specified, open the workbook from a file,  if :stream is specified, open the workbook from an input stream, otherwise return a new workbook"
  [& {:keys [file stream] :as opts}]
  (if (contains? opts :file)
    (with-open [fis (FileInputStream. file)]
      (WorkbookFactory/create fis))
    (if (contains? opts :stream)
      (WorkbookFactory/create stream)
      (HSSFWorkbook.))))

(defn save-as
  "save the workbook"
  [wb f]
  (with-open [fos (FileOutputStream. f)]
    (.write wb fos)))

(defn worksheet
  "return a handle to a worksheet.  if :create is specified, create a new worksheet"
  [wb name & opts]
  (let [opts (apply hash-set opts)]
    (if (contains? opts :create)
      (.createSheet wb (WorkbookUtil/createSafeSheetName name))
      (.getSheet wb name))))

(defn cell-type
  "return the type of a cell as a clojure keyword"
  [cell]
  (let [types {Cell/CELL_TYPE_BLANK :blank
               Cell/CELL_TYPE_BOOLEAN :boolean
               Cell/CELL_TYPE_ERROR :error
               Cell/CELL_TYPE_FORMULA :fomula
               Cell/CELL_TYPE_NUMERIC :numeric
               Cell/CELL_TYPE_STRING :string}]
    (get types (.getCellType cell))))

(defn cell-value
  "return the value of a cell as a clojure object"
  [cell & {es :empty-string}]
  (case (cell-type cell)
        :blank (if es
                 ""
                 nil)
        :boolean (.getBooleanCellValue cell)
        :error (.getStringCellValue cell)
        :formula (.getStringCellValue cell)
        :numeric (.getNumericCellValue cell)
        :string (.getStringCellValue cell)))

(defn row-seq
  "iterate over the rows in a worksheet"
  [ws]
  (iterator-seq (.rowIterator ws)))

(defn as-table
  "return worksheet as a table.  see jlk.table."
  [ws]
  (let [dta (row-seq ws)
        headers (map keyword (map cell-value (first dta)))
        rows (rest dta)]
    [:columns (map (fn [h] {:key h :text h}) headers)
     :rows (map (fn [row] (zipmap headers (map cell-value row))) rows)]))

;; (defn new-worksheet
;;   "returns a reference to the new worksheet"
;;   [wb name]
;;   (.createSheet wb (WorkbookUtil/createSafeSheetName name)))

;; (defn worksheet
;;   [wb idx]
;;   (.getSheetAt wb idx))

;; (defn default-column-width
;;   [ws]
;;   (.getDefaultColumnWidth ws))

;; (defn default-row-height
;;   "in points"
;;   [ws]
;;   (.getDefaultRowHeightInPoints ws))

;; (declare new-cell)
;; (defn new-row
;;   "ws - worksheet, ridx - row index (integer, 0 based), optional values (seq)"
;;   ([ws ridx]
;;      (.createRow ws ridx))
;;   ([ws ridx values]
;;      (let [row (new-row ws ridx)]
;;        (doseq [[cidx value] (partition 2 (interleave (range (count values)) values))]
;;          (new-cell row cidx value))
;;        row)))

;; (defn row
;;   [ws ridx]
;;   (.getRow ws ridx))

;; (defn row-index
;;   [row]
;;   (.getRowNum row))

;; (defn row-height
;;   "in points"
;;   [row]
;;   (.getHeightInPoints row))

;; (defn row-last-index
;;   "return the column index of the last cell in row"
;;   [row]
;;   (.getLastCellNum row))

;; (defn new-cell
;;   "note value cannot be an integer (use double)"
;;   [row cidx value]
;;   (doto (.createCell row cidx)
;;     (.setCellValue value)))

;; (defn cell
;;   [row cidx]
;;   (.getCell row cidx))

;; (defn cell-seq
;;   "create a sequence of all cells in a row"
;;   ;; fixme - this should probably be made lazy
;;   [row]
;;   (iterator-seq (.cellIterator row)))

;; (defn cell-row-index
;;   [cell]
;;   (.getRowIndex cell))

;; (defn cell-column-index
;;   [cell]
;;   (.getColumnIndex cell))

;; (defn cell-style
;;   [cell]
;;   (.getCellStyle cell))

;; (defn cell-alignment
;;   [cell]
;;   (let [halign {CellStyle/ALIGN_GENERAL :general
;;                 CellStyle/ALIGN_LEFT :left
;;                 CellStyle/ALIGN_CENTER :center
;;                 CellStyle/ALIGN_RIGHT :right
;;                 CellStyle/ALIGN_FILL :fill
;;                 CellStyle/ALIGN_JUSTIFY :justify
;;                 CellStyle/ALIGN_CENTER_SELECTION :center_selection}
;;         valign {CellStyle/VERTICAL_BOTTOM :bottom
;;                 CellStyle/VERTICAL_CENTER :center
;;                 CellStyle/VERTICAL_JUSTIFY :justify
;;                 CellStyle/VERTICAL_TOP :top}
;;         h (get halign (.getAlignment (cell-style cell)))
;;         v (get valign (.getVerticalAlignment (cell-style cell)))]
;;     [h v]))

;; (defn cell-rotation
;;   [cell]
;;   (.getRotation (cell-style cell)))

;; (defn column-width
;;   "1/256 of a character width"
;;   [ws colidx]
;;   (.getColumnWidth ws colidx))

;; (defn merged-regions-count
;;   [ws]
;;   (.getNumMergedRegions ws))

;; (defn merged-region
;;   [ws idx]
;;   (.getMergedRegion ws idx))

;; (defn merged-region-seq
;;   [ws]
;;   (map #(merged-region ws %) (range (merged-regions-count ws))))

;; (defn merged-region-bounds
;;   [mr]
;;   [(.getFirstRow mr)
;;    (.getLastRow mr)
;;    (.getFirstColumn mr)
;;    (.getLastColumn mr)
;;    ])
