(ns jlk.log.core
  (:use [jlk.utility.core :only [exception bulkdefn]]
        [jlk.system.core :only [register-shutdown-hook]]))

;;
;; level definition and comparison
;;

(def ^:dynamic *levels*
  {:all 0
   :none 256
   :trace 1
   :debug 2
   :info 4
   :warn 8
   :error 16
   :fatal 32
   :probe 64})

(bulkdefn "direct level comparisons"
          [l1 l2]
          l>= (>= (get *levels* l1) (get *levels* l2))
          l<= (<= (get *levels* l1) (get *levels* l2))
          l> (> (get *levels* l1) (get *levels* l2))
          l< (< (get *levels* l1) (get *levels* l2))
          l= (= (get *levels* l1) (get *levels* l2)))

;;
;; logger system configuration and state management helpers
;;

(defrecord Subsystem [name enabled?])

(defonce ^:dynamic *level* (atom :info))
(defonce ^:dynamic *subsystems* (atom {}))
(defonce ^:dynamic *loggers* (atom []))
(defonce ^:dynamic *protected-writers* (atom (reduce conj #{} [*out* *err*])))

(defn set-level!
  "set level of logging to level (:info by default)"
  ([]
     (set-level! :info))
  ([level]
     (reset! *level* level)))

(defn shutdown-loggers!
  "call .close on the writers of all loggers, unless defined in *protected-writers* (eg. *out* *err*)"
  []
  (swap! *loggers* (fn [loggers]
                     (doseq [{:keys [writer]} loggers]
                       (if-not (contains? @*protected-writers* writer)
                         (try (.close writer)
                              (catch Exception e (println "error shutting down logger")))))
                     [])))

;; try to behave - call .close on open writers
(defonce ^:dynamic *register-shutdown*
  (do (register-shutdown-hook (fn [] (println "shutting down logging...") (shutdown-loggers!)))
      :registered))

(defn set-logger!
  "set the current logger to logger, calling shutdown-loggers! on all other defined loggers.  see jlk.log.loggers"
  [logger]
  (shutdown-loggers!)
  (reset! *loggers* [logger]))

(defn add-logger!
  "add another logger.  see jlk.log.loggers"
  [logger]
  (swap! *loggers* conj logger))

(defmacro enable-subsystem!
  "set the enabled flag on subsystem (defaults to current ns)"
  ([]
     `(enable-subsystem! ~(str *ns*)))
  ([subsystem]
     `(if (get @*subsystems* ~subsystem)
        (swap! *subsystems* assoc-in [~subsystem :enabled?] true))))

(defmacro disable-subsystem!
  "unset the enabled flag on subsystem (defaults to current ns)"
  ([]
     `(disable-subsystem! ~(str *ns*)))
  ([subsystem]
     `(if (get @*subsystems* ~subsystem)
        (swap! *subsystems* assoc-in [~subsystem :enabled?] false))))

(defmacro subsystem-enabled?
  "query if subsystem is enabled (defaults to current ns)"
  ([]
     `(subsystem-enabled? ~(str *ns*)))
  ([subsystem]
     `(get-in @*subsystems* [~subsystem :enabled?] false)))

(defmacro get-or-create-subsystem!
  "get the Subsystem record for subsystem (defaults to current ns).  will create if doesn't exist.  defaults to :enabled? false"
  ([]
     `(get-or-create-subsystem! ~(str *ns*)))
  ([subsystem]
     `(if-let [rv# (get @*subsystems* ~subsystem)]
       rv#
       (let [ss# (Subsystem. ~subsystem false)]
         (swap! *subsystems* assoc ~subsystem ss#)
         ss#))))

;;
;; actual log function
;;

(defmacro log
  "system logging

the log system is set up with various subsystems and levels along with zero or more loggers.  subsystems are typically the namespace in which the log function is called.

loggers define how the output is handled, eg. to the console, to a file, etc.
a logger is added to the system using set-logger! or add-logger!
a number of loggers have been pre-defined in jlk.log.logging

subsystems are added automatically and set to the current namespaces
each subsystem can be enabled or disabled using enable-subsystem! or disable-subsystem!

the log level is set to :info by default and applies accross all enabled subsystems.
other levels are :fatal, :error, :warn, :info, :debug, :trace
the level of logging is set using set-level!

helper functions are defined for all the levels: fatal, error, warn, info, debug, trace

logging functions are printed using printf semantics for the string s & args"
  [level s & args]
  `(let [ss# (get-or-create-subsystem!)]
     (if (:enabled? ss#)
       (if (l>= ~level @*level*)
         (doseq [logger# @*loggers*]
           (let [w# (:writer logger#)]
             (.write w# ((:log-fn logger#) (:name ss#) ~level (format ~s ~@args)))
             (if (:flush-on-write logger#)
               (.flush w#))))))))

;;
;; logger helpers
;; 
;; note we can't define a default subsystem because we would end up with multiple
;; variadic overloads
;;

(defmacro fatal
  "see jlk.log.core/log"
  [s & args]
  `(log :fatal ~s ~@args))

(defmacro error
  "see jlk.log.core/log"
  [s & args]
  `(log :error ~s ~@args))

(defmacro warn
  "see jlk.log.core/log"
  [s & args]
  `(log :warn ~s ~@args))

(defmacro info
  "see jlk.log.core/log"
  [s & args]
  `(log :info ~s ~@args))

(defmacro debug
  "see jlk.log.core/log"
  [s & args]
  `(log :debug ~s ~@args))

(defmacro trace
  "see jlk.log.core/log"
  [s & args]
  `(log :trace ~s ~@args))

;; replace probe with clojure.tools.trace?
