(ns jlk.log.loggers
  (:use [jlk.log.core :only [l>= l<= l> l< l=]]
        [jlk.system.core :only [current-thread]]
        [jlk.time.core :only [now convert formatter]]
        [clojure.java.io :only [writer]]))

;;
;; Various loggers or define your own
;; TODO - implement some known logging protocol? look at apache?
;;

(defrecord Logger [writer flush-on-write log-fn])
(defn logger [writer flush-on-write log-fn]
  (Logger. writer flush-on-write log-fn))

(defn basic-console-logger
  "very basic, log to the console with no context"
  []
  (logger *out*
          false
          (fn [subsystem level s] (format "%s\n" s))))

(defn detailed-console-logger
  "log to the console, provide a bit more detail"
  []
  (logger *out*
          false
          (fn [subsystem level s] (format "[%s %s] %s\n" (.toUpperCase (name level)) subsystem s))))

(defn slow-console-logger
  "log to the console, flush each output"
  []
  (logger *out*
          true
          (fn [subsystem level s] (format "[%s %s] %s\n" (.toUpperCase (name level)) subsystem s))))

(defn fancy-console-logger
  "log to the console, providing visual clues for higher levels, timing and thread context information"
  []
  (logger *out*
          false
          (fn [subsystem level s]
            (format (if (l>= level :error)
                        "** %5.5s : %s ** [ %s ] %s\n%s\n"
                        (if (l>= level :warn)
                          "!! %5.5s : %s !! [ %s ] %s\n%s\n"
                          "-- %5.5s : %s -- [ %s ] %s\n%s\n"))
                    (.toUpperCase (name level))
                    subsystem
                    (convert (now) (formatter :iso))
                    (str (current-thread))
                    s))))

(defn file-logger
  "log to a file"
  [filename]
  (logger (writer filename :append true)
          true
          (fn [subsystem level s]
            (format (if (l>= level :error)
                        "** %5.5s : %s ** [ %s ] %s\n%s\n"
                        (if (l>= level :warn)
                          "!! %5.5s : %s !! [ %s ] %s\n%s\n"
                          "-- %5.5s : %s -- [ %s ] %s\n%s\n"))
                    (.toUpperCase (name level))
                    subsystem
                    (convert (now) (formatter :iso))
                    (str (current-thread))
                    s))))
