(ns clj-http.headers
  (:use [clj-http.util :only [lower-case]])
  (:require [clojure.string :as s]
            [potemkin :as potemkin]))

(def special-cases ["Content-MD5"
                    "DNT"
                    "ETag"
                    "P3P"
                    "TE"
                    "WWW-Authenticate"
                    "X-ATT-DeviceId"
                    "X-UA-Compatible"
                    "X-WebKit-CSP"
                    "X-XSS-Protection"])

(defn special-case [#^String s]
  (or (first (filter #(.equalsIgnoreCase #^String % s) special-cases))
      s))

(defn title-case [#^Character c]
  (Character/toTitleCase c))

(defn canonicalize [h]
  (if (string? h)
    h
    (-> (name h)
        (lower-case)
        (s/replace #"(?:^.|-.)"
                   (fn [s]
                     (if (next s)
                       (str (first s)
                            (title-case (second s)))
                       (str (title-case (first s))))))
        (special-case))))

(defn normalize [k]
  (lower-case (name k)))

(potemkin/def-map-type HeaderMap [m]
  (get [_ k v]
       (second (get m (normalize k) [nil v])))
  (assoc [_ k v]
    (HeaderMap. (assoc m (normalize k) [(canonicalize k) v])))
  (dissoc [_ k]
          (HeaderMap. (dissoc m (normalize k))))
  (keys [_]
        (map first (vals m)))
  clojure.lang.Associative
  (containsKey [_ k]
               (contains? m (normalize k))))

(defn header-map
  " Returns a new header map with supplied mappings."
  [& keyvals]
  (into (HeaderMap. {})
        (apply array-map keyvals)))

(defn wrap-header-map
  "Middleware that converts headers from a map into a header-map."
  [client]
  (fn [req]
    (let [raw-headers (:raw-headers req)
          req-headers (:headers req)
          req (if req-headers
                (-> req (assoc :headers (into (header-map) req-headers)))
                req)
          resp (client (assoc req :raw-headers true))
          resp-headers (:raw-headers resp)
          resp (if resp-headers
                 (-> resp (assoc :headers (into (header-map) resp-headers)))
                 resp)]
      (if raw-headers
        resp
        (dissoc resp :raw-headers)))))
