;;This is a quick hack around the hidpi scaling issues....
(ns nightcode.scaling          
  (:import [java.awt Toolkit Font]
           [javax.swing Icon UIManager UIDefaults]
           [javax.swing.plaf FontUIResource IconUIResource]))
;;adapted from https://stackoverflow.com/questions/38169983/hidpi-support-in-java-swing-for-multiple-look-and-feels 
(defprotocol ITweaker
  (initial-tweaks [this])
  (modify-font [this k f])
  (modify-icon [this k i])
  (modify-integer [this k i])) 

(defn ends-with [tgt s]
  (let [re (re-pattern (str ".*" tgt "$"))]
    (re-find re  s )))

(defn ends-with-some [xs s]
  (some (fn [x]
          (ends-with x s)) xs))

(defmacro check [t  body]
  `(let [res# ~body]
     (assert (instance? ~t res#) (str [:wrong-type ~t res# ~body]))
     res#))

(defrecord basic-tweaker
    [scalefactor ^UIDefaults uidefaults initial on-font on-icon on-int ]
  ITweaker
  (initial-tweaks [this] (if initial
                           (do (initial this) this)
                           this))
  (modify-font [this k f] (check java.awt.Font (on-font k f scalefactor)))
  (modify-icon [this k i] (on-icon this k i))
  (modify-integer [this k i] (check java.lang.Integer (on-int k i scalefactor))) 
  )

;; protected static FontUIResource newScaledFontUIResource(Font original, float scale) {
;;   int newSize = Math.round(original.getSize() * scale);
;;   return new FontUIResource(original.getName(), original.getStyle(), newSize);
;;                                                                                      }

(defn new-scaled-font-ui-resource [^Font original scale]
  (let [newsize (Math/round (* (.getSize original) scale))]
    (FontUIResource. (.getName original) (.getStyle original) newsize)))

;;Ignores title & accelerator fonts (for example)
(defn ^Font modify-font! [k ^Font original scale-factor]
  (if (and (instance? FontUIResource original)
           (ends-with ".font" (str k)))
    (new-scaled-font-ui-resource original scale-factor)
    original))

;; public Icon modifyIcon(Object key, Icon original) {
;;   return new IconUIResource(new ScaledIcon(original, scaleFactor));
;;                                                    }

;;scaled icon doesn't exist...
(defn ^Icon modify-icon! [k ^Icon original scale]
  original
  #_(IconUIResource. (ScaledIcon. original scale)))


;; public Integer modifyInteger(Object key, Integer original) {
;;   if (!endsWithOneOf(lower(key), LOWER_SUFFIXES_FOR_SCALED_INTEGERS)) {
;;     return original;
;;   }
;;   return (int) (original * scaleFactor);
                                                             
(def int-scales ["width", "height", "indent", "size", "gap"])

(defn lower [x] (if (string? x)
                  (clojure.string/lower-case x)
                  ""))

(defn ^int modify-integer! [k original scale-factor]
  (if (not (ends-with-some int-scales (lower k)))
    original
    (int (* original scale-factor))))


(defn tweaker [scalefactor & {:keys [init on-font on-icon on-int]}]
  (->basic-tweaker
   scalefactor
   (UIManager/getLookAndFeelDefaults)
   (or init    (fn [this] this))
   (or on-font (fn [this k original] (modify-font! k original scalefactor))) 
   (or on-icon (fn [this k original] (modify-icon! k original scalefactor)))
   (or on-int  (fn [this k original] (modify-integer! k original scalefactor)))))

;; private static float getCurrentScaling() {
;;   int dpi = Toolkit.getDefaultToolkit().getScreenResolution();
;;   return dpi / 96f;

(defn get-current-scaling []
  (let [dpi (.getScreenResolution (Toolkit/getDefaultToolkit))]
    (/ dpi 96.0)))

;;Windows already scales fonts, scrollbar sizes (etc) according to the system DPI settings.
;;This lets us adjust to the REQUESTED scale factor, relative to the CURRENT scale factor
(defn ->windows-tweaker [scalefactor ]
  (tweaker (double (/ scalefactor (get-current-scaling)))
           :on-icon (fn [this k original scale]  original )))


;; // Setting "defaultFont" above is sufficient as this will be inherited by all others
;; public Font modifyFont(Object key, Font original) {
;;   return original;
;; }

;;Scaling Radio or CheckBox button icons leads to really weird artifacts in Nimbus? Disable
(defn ->nimbus-tweaker [scalefactor]
  (tweaker scalefactor
   :init (fn [this]
           (let [^UIDefaults defaults (UIManager/getLookAndFeelDefaults)
                 ^Font font  (.getFont defaults "defaultFont")]
             (do (when font
                   (.put defaults "defaultFont"
                         (FontUIResource.
                          (.getName font)
                          (.getStyle font)
                          (Math/round (float (* (.getSize font) scalefactor))))))
                 this)))
   :on-font (fn [k v scale] v)
   :icon    (fn [k v scale] v)))

;;API for setting everything up.
(defn create-tweaker! [dpi-scale]
  (let [test (->> (UIManager/getLookAndFeel) (.getName) (.toLowerCase))]
    (cond (.contains test "windows") (->windows-tweaker dpi-scale)
          (.contains test "nimbus")  (->nimbus-tweaker dpi-scale)
          :else (tweaker dpi-scale))))

(defn get-updated-value [tweaker k original]
  (cond (instance? Font original) (modify-font tweaker k original)
        (instance? Icon original) (modify-icon tweaker k original)
        (instance? Integer original) (modify-integer tweaker k original)
        :else nil))

(def prior (atom nil))
(defn tweak-ui-defaults! [tweaker multiplier]
  (let [^UIDefaults defaults (UIManager/getLookAndFeelDefaults)
        _                    (initial-tweaks tweaker)]
    (doseq [[k v] (seq defaults)]
      (let [original (get defaults k)
            _ (reset! prior [k v original])
            new-val  (get-updated-value tweaker k original)
                         ]
        (when (and new-val (not= new-val original))
          (.put defaults k new-val))))))

(defn scale-ui-defaults! []
  (let [dpi-scale (-> (Toolkit/getDefaultToolkit) (.getScreenResolution))
        tweaker  (create-tweaker! dpi-scale)]
    (tweak-ui-defaults! tweaker dpi-scale)))
