(ns repl-gui.core
  (:require [seesaw.core :as sc]
            [seesaw.table :as st])
  (:import (java.util Date)
           (java.text SimpleDateFormat)
           (javax.swing JTable)
           (java.awt.event MouseEvent)))

(defmulti data->table-model
          (fn [data-rows]
            (cond
              (or (set? data-rows) (sequential? data-rows)) :sequence
              (map? data-rows) :map
              :else :unknown)))

(defmethod data->table-model :map [key->value]
  (let [rows (map #(vector % (get key->value %)) (keys key->value))]
    {:columns ["property" "value"] :rows rows}))

(defmethod data->table-model :sequence [items]
  (let [items (vec items)]
    (cond
      (every? map? items)
      (let [ks (distinct (mapcat keys items))]
        {:columns ks
         :rows (map (fn [item]
                      (map #(get item %) ks))
                    items)})

      (every? sequential? items)
      (let [ks (range (apply max (map count items)))]
        {:columns ks
         :rows (map (fn [item]
                      (map #(get item %) ks))
                    items)})

      :else
      {:columns ["index" "item"]
       :rows (map-indexed vector items)})))

(defmethod data->table-model :unknown [item]
  (data->table-model (bean item)))

(defn show! [component]
  (-> (sc/frame :title "Data" :content (sc/scrollable component)) (sc/pack!) (sc/show!)))

(defn data-table
  ([data] (data-table data {}))
  ([data {:keys [column-labels]}]
   (let [{:keys [rows columns]} (data->table-model data)
         columns (mapv #(if (keyword? %) % (keyword (str %))) (or column-labels columns))
         rows (mapv vec rows)
         ^JTable table (sc/table :model [:columns columns :rows rows])]
     (sc/listen table
                :mouse-clicked
                (fn [^MouseEvent e]
                  (let [row-index (.rowAtPoint table (.getPoint e))
                        column-index (.columnAtPoint table (.getPoint e))]
                    (when (and (not (neg? row-index))
                               (not (neg? column-index)))
                      (let [row-data (st/value-at table row-index)
                            column-key (nth columns column-index)
                            cell-data (get row-data column-key)]
                        (when (and (not (nil? cell-data))
                                   (not (number? cell-data))
                                   (not (string? cell-data))
                                   (not (keyword? cell-data)))
                          (show! (data-table cell-data))))))))
     table)))

(def standard-date-format (new SimpleDateFormat "yyyy-MM-dd HH:mm:ss.SSS"))
(defn format-date [date]
  (.format standard-date-format date))

(defn watcher
  ([reference] (watcher reference {}))
  ([reference {:keys [limit elide-duplicates?]
               :or {limit 0
                    elide-duplicates? false}}]
   (let [table (data-table [{:timestamp (format-date (new Date)) :state @reference}])]
     (add-watch reference
                :ui-watcher
                (fn [_ _ old-state new-state]
                  (when-not (and elide-duplicates? (= old-state new-state))
                    (let [row-count (st/row-count table)]
                      (st/insert-at! table row-count {:timestamp (format-date (new Date)) :state new-state})
                      (when (and (pos? limit) (>= row-count limit))
                        (dotimes [i (inc (- row-count limit))]
                          (st/remove-at! table i)))
                      (sc/scroll! table :to :bottom)))))
     table)))

(comment

  (show! (data-table [[1 "jon" 29]
                      [2 "jon2" {:test "nested" :hmmm [1 2 3]}]
                      [3 "jon3" 999]
                      [3 "jon3" 999 888]]
                     {:column-labels ["id" "name" "something" "optional"]}))

  (show! (data-table [[1 "jon" 29]
                      [2 "jon2" {:test "nested" :hmmm [1 2 3]}]
                      [3 "jon3" 999]
                      {:jello "furled"}
                      [3 "jon3" 999 888]]))

  (do
    (def x (atom [1]))
    (show! (watcher x {:limit 0 :elide-duplicates? true}))
    (swap! x conj (rand-int 100))
    (swap! x conj (rand-int 100))
    (swap! x conj (rand-int 100))
    (swap! x identity))

  (show! (data-table [(new Date) (new Date) (new Date)]))

  (show! (data-table (new Date)))

  (show! (data-table #{1 2 3}))

  (show! (data-table (bean (new Date))))

  (show! (data-table [1 2 3 4 5 6 7 8]))

  (show! (data-table [{:id 1 :name "jon" :age 29}
                      {:id 1 :name "jon2" :age 103}
                      {:id 1 :name "jon3" :age 999}]))


  (show! (data-table [{:id 1 :name "jon" :age 29 :meh "hi"}
                      {:id 1 :name "jon2" :age 103 :hi "meh"}
                      {:id 1 :name "jon3" :age 999}]))

  )
