(ns repl-gui.core
  (:require [seesaw.core :as sc]
            [seesaw.table :as st])
  (:import (java.util Date UUID)
           (java.text SimpleDateFormat)
           (javax.swing JTable)
           (java.awt.event MouseEvent)))

(defn seq-ish? [x] (or (set? x)
                       (sequential? x)
                       (.isArray (class x))))

(defmulti data->table-model
          (fn [data]
            (cond
              (map? data) :map
              (and (seq-ish? data) (every? map? data)) :maps
              (and (seq-ish? data) (every? seq-ish? data)) :colls
              (seq-ish? data) :coll
              :else :bean)))

(defmethod data->table-model :map [key->value]
  (let [rows (map #(vector % (get key->value %)) (keys key->value))]
    {:columns ["property" "value"] :rows rows}))

(defmethod data->table-model :maps [maps]
  (let [maps (vec maps)
        ks (distinct (mapcat keys maps))]
    {:columns ks
     :rows (map (fn [item]
                  (map #(get item %) ks))
                maps)}))

(defmethod data->table-model :colls [colls]
  (let [colls (vec colls)
        ks (range (apply max (map count colls)))]
    {:columns ks
     :rows (map (fn [item]
                  (map #(get item %) ks))
                colls)}))

(defmethod data->table-model :coll [items]
  {:columns ["index" "item"]
   :rows (map-indexed vector items)})

(defmethod data->table-model :bean [item]
  (data->table-model (bean item)))

(defn show! [{:keys [component on-close]}]
  (let [frame (sc/frame :title "Data"
                        :on-close :dispose
                        :content (sc/scrollable component))]
    (when on-close
      (sc/listen frame :window-closed on-close))
    (-> frame (sc/pack!) (sc/show!))))

(defn data-table
  ([data] (data-table data {}))
  ([data {:keys [column-labels]}]
   (let [{:keys [rows columns]} (data->table-model data)
         columns (mapv #(if (keyword? %) % (keyword (str %))) (or column-labels columns))
         rows (mapv vec rows)
         ^JTable table (sc/table :model [:columns columns :rows rows])]
     (sc/listen table
                :mouse-clicked
                (fn [^MouseEvent e]
                  (let [row-index (.rowAtPoint table (.getPoint e))
                        column-index (.columnAtPoint table (.getPoint e))]
                    (when (and (not (neg? row-index))
                               (not (neg? column-index)))
                      (let [row-data (st/value-at table row-index)
                            column-key (nth columns column-index)
                            cell-data (get row-data column-key)]
                        (when (and (not (nil? cell-data))
                                   (not (number? cell-data))
                                   (not (string? cell-data))
                                   (not (keyword? cell-data)))
                          (show! (data-table cell-data))))))))
     {:component table})))

(def standard-date-format (new SimpleDateFormat "yyyy-MM-dd HH:mm:ss.SSS"))
(defn format-date [date]
  (.format standard-date-format date))

(defn watcher
  ([reference] (watcher reference {}))
  ([reference {:keys [limit elide-duplicates?]
               :or {limit 0
                    elide-duplicates? false}}]
   (let [{table :component} (data-table [{:timestamp (format-date (new Date)) :state @reference}])
         watch-key (str (UUID/randomUUID))
         on-close (fn [_] (remove-watch reference watch-key))]
     (add-watch reference
                watch-key
                (fn [_ _ old-state new-state]
                  (when-not (and elide-duplicates? (= old-state new-state))
                    (let [row-count (st/row-count table)]
                      (st/insert-at! table row-count {:timestamp (format-date (new Date)) :state new-state})
                      (when (and (pos? limit) (>= row-count limit))
                        (dotimes [i (inc (- row-count limit))]
                          (st/remove-at! table i)))
                      (sc/scroll! table :to :bottom)))))
     {:component table :on-close on-close})))

(comment

  (show! (data-table [[1 "jon" 29]
                      [2 "jon2" {:test "nested" :hmmm [1 2 3]}]
                      [3 "jon3" 999]
                      [3 "jon3" 999 888]]
                     {:column-labels ["id" "name" "something" "optional"]}))

  (show! (data-table [[1 "jon" 29]
                      [2 "jon2" {:test "nested" :hmmm [1 2 3]}]
                      [3 "jon3" 999]
                      {:jello "furled"}
                      [3 "jon3" 999 888]]))

  (do
    (def x (atom [1]))
    (show! (watcher x {:limit 0 :elide-duplicates? true}))
    (swap! x conj (rand-int 100))
    (swap! x conj (rand-int 100))
    (swap! x conj (rand-int 100))
    (swap! x identity))

  (show! (data-table [(new Date) (new Date) (new Date)]))

  (show! (data-table (new Date)))

  (show! (data-table #{1 2 3}))

  (show! (data-table (bean (new Date))))

  (show! (data-table [1 2 3 4 5 6 7 8]))

  (show! (data-table [{:id 1 :name "jon" :age 29}
                      {:id 1 :name "jon2" :age 103}
                      {:id 1 :name "jon3" :age 999}]))


  (show! (data-table [{:id 1 :name "jon" :age 29 :meh "hi"}
                      {:id 1 :name "jon2" :age 103 :hi "meh"}
                      {:id 1 :name "jon3" :age 999}]))

  )
