(ns jtk-dvlp.re-frame.tasks
  (:require
   [cljs.core.async]
   [jtk-dvlp.async :as a]
   [re-frame.core :as rf]
   [re-frame.interceptor :as interceptor]))


;; IDEA: Einen Task auch eine cancel-fn mit geben, um den prozess abzubrechen.
;; IDEA: Einen Task abbrechen können, mindestens mal den Merge in die DB oder das Ausführen eines Events bei acofxs.

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Functions

(defn register
  [db {:keys [::id] :as task}]
  (assoc-in db [::db :tasks id] task))

(defn unregister
  [db {:keys [::id]}]
  (update-in db [::db :tasks] dissoc id))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Interceptors

(defn- fx-handler-run?
  [{:keys [stack]}]
  (->> stack
       (filter #(= :fx-handler (:id %)))
       (seq)))

(defn- normalize-task
  [name-or-task]
  (if (map? name-or-task)
    name-or-task
    {::name name-or-task}))

(defn- normalize-fxs
  [fxs]
  (for [fx fxs]
    (cond
      (keyword? fx)
      {:effect-key fx
       :completion-keys #{:on-completed}}

      (vector? fx)
      {:effect-key (first fx)
       :completion-keys (into #{} (rest fx))}

      :else fx)))

(defonce ^:private !task<->fxs-counters
  (atom {}))

(defn- unregister-by-fx
  [effect completion-keys task]
  (reduce
   (fn [effect completion-key]
     (update effect completion-key (partial vector ::unregister-and-dispatch-original task)))
   effect
   completion-keys))

(defn- unregister-by-fxs
  [context {:keys [::id] :as task} fxs]
  (swap! !task<->fxs-counters assoc id (count fxs))
  (reduce
   (fn [context {:keys [effect-key completion-keys]}]
     (interceptor/update-effect context effect-key unregister-by-fx completion-keys task))
   context
   fxs))

(defn- unregister-by-failed-acofx
  [context task ?acofx]
  (cljs.core.async/take!
   ?acofx
   (fn [result]
     (when (a/exception? result)
       (rf/dispatch [::unregister task]))))
  context)

(defn- get-db
  [context]
  (or
   (interceptor/get-effect context :db)
   (interceptor/get-coeffect context :db)))

(defn- includes-acofxs?
  [context]
  (contains? context :acoeffects))

(defn- handle-acofx-variant
  [{:keys [acoeffects] :as context} task fxs]
  (let [db
        (get-db context)

        {:keys [dispatch-id ?error]}
        acoeffects

        task
        (assoc task ::id dispatch-id)]

    (if (fx-handler-run? context)
      (if (seq fxs)
        (unregister-by-fxs context task fxs)
        (interceptor/assoc-effect context :db (unregister db task)))
      (-> context
          (interceptor/assoc-effect :db (register db task))
          (unregister-by-failed-acofx task ?error)))))

(defn- handle-straight-variant
  [context task fxs]
  (let [db
        (get-db context)

        task
        (assoc task ::id (random-uuid))]

    ;; NOTE: no need to register task in every case. the task register
    ;;       would be effectiv too late after finish the handler.
    (cond-> context
      (seq fxs)
      (-> (interceptor/assoc-effect :db (register db task))
          (unregister-by-fxs task fxs)))))

(defn as-task
  ([name-or-task]
   (as-task name-or-task nil))

  ([name-or-task fxs]
   (let [task
         (normalize-task name-or-task)

         fxs
         (normalize-fxs fxs)]

     (rf/->interceptor
      :id
      :as-task

      :after
      (fn [context]
        (cond
          (includes-acofxs? context)
          (handle-acofx-variant context task fxs)

          :else
          (handle-straight-variant context task fxs)))))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Events

(rf/reg-event-db ::register
  (fn [db [_ task]]
    (register db task)))

(rf/reg-event-db ::unregister
  (fn [db [_ task]]
    (unregister db task)))

(rf/reg-event-fx ::unregister-and-dispatch-original
  (fn [_ [_ task original-event-vec & original-event-args]]
    {::unregister-and-dispatch-original [task original-event-vec original-event-args]}))

(rf/reg-fx ::unregister-and-dispatch-original
  (fn [[{:keys [::id] :as task} original-event-vec original-event-args]]
    (when original-event-vec
      (rf/dispatch (into original-event-vec original-event-args)))

    (if (= 1 (get @!task<->fxs-counters id))
      (do
        (swap! !task<->fxs-counters dissoc id)
        (rf/dispatch [::unregister task]))
      (swap! !task<->fxs-counters update id dec))))


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Subscriptions

(rf/reg-sub ::db
  (fn [{:keys [::db]}]
    db))

(rf/reg-sub ::tasks
  :<- [::db]
  (fn [{:keys [tasks]}]
    tasks))

(rf/reg-sub ::running?
  :<- [::tasks]
  (fn [tasks [_ name]]
    (if name
      (->> tasks (vals) (filter #(= (::name %) name)) (first) (some?))
      (-> tasks (seq) (some?)))))
