# WebSocket Server

This is the server side of the websocket connection.  This is meant to
be paired with [fentontravers/websocket-client](https://github.com/fentontravers/websocket-client).

# Clojars

<a href="https://clojars.org/juleffel/websocket-server/" target="_blank">![Foo](https://clojars.org/juleffel/websocket-server/latest-version.svg)</a>

# Usage

```clojure
(require '[websocket-server.core :refer :all])

(defn request-handler-upcase-string
  "The function that will take incoming data off the websocket,
  process it and return a reponse.  In our case we'll simply UPPERCASE
  whatever is received."
  [data] (clojure.string/upper-case (str data)))

; Always call this module functions with a port, unless you want to apply it to every server you opened on all ports.
(def port 8899)

(defn start
  "Demonstrate how to use the websocket server library."
  []
  (start-ws-server port :on-receive request-handler-upcase-string))

(defn send-all!
  [data]
  (send-all! port data))

(defn stop
  "Stop websocket server"
  []
  (stop-ws-server port))
```
  
Here is another example that expects EDN in the form of a map that
looks like `{:count 1}`, or just a map with a key `:count` and some
integer value.  Then it increments that value by 10 and returns it
back.

```clojure
(defn request-handler-add10 
  [data]
  (->> data
       edn/read-string
       :count
       (+ 10)
       (hash-map :count)
       str))
```

# Optional arguments

- `:on-open (fn [send] (send "Data"))` : on-open function is called at establishment of a
new connection with a `send` function you can use to send data to this channel.
- `:on-receive (fn [data] (do-something data))` : called when a message is received.
Send the result of the on-receive function to the initiating channel.
- `:on-close (fn [status] (println "Channel close with status:" status))` : Called when a connection with a client closed.
- `:in-fn (fn [data] (parse data))` : Applied each time server receive a message, before sending it to other handlers.
- `:out-fn (fn [data] (format data))` : Applied each time server send a message.

`:on-event` functions all have a `:on-event-full` version with three more arguments:
`port channel req` for advanced users. `:on-receive-full` doesn't send back result to server in
this version, it offers the `send` function as first argument, like on-open.

# Multiple servers usage

```clojure
(start-ws-server 8000 :on-receive request-handler-1)
(start-ws-server 8001 :on-receive request-handler-2)
(start-ws-server 8002 :on-receive request-handler-3)

; Send "Hello" to all channels opened to websocket on port 8000
(send-all! 8000 "Hello")

(stop-ws-server 8002)

; Send "Hi!" to all channels opened to websocket on port 8000 or 8001
(send-all! "Hello")

(stop-all-ws-servers)
```

# Use transformer functions when receiving and sending data

```clojure
(require '[clojure.data.json :as json])

(defn request-handler-json
    [[action data]]
    [action
     (case action
        "upcase" (request-handler-upcase-string data)
        data)])

; json/read-str will be applied before sending data to request-json-handler
; json/write-str will be applied before sending data from request-json-handler
;   back on the websocket, or when using send-all!
(start-ws-server 8000
  :on-receive request-handler-json
  :in-fn json/read-str
  :out-fn json/write-str)
```
