(ns hbase-clj.admin
  "Set of functions for administrative tasks, such as creating, dropping,
  and altering tables. Also provides functions for monitoring the cluster
  status."
  (:require [clojure.reflect :as refl]
            [clojure.string :as str]
            [hbase-clj.util :refer [->bytes bytes-> table-name-for empty-bytes]])
  (:import [org.apache.hadoop.hbase
            util.Bytes
            io.compress.Compression$Algorithm
            io.encoding.DataBlockEncoding
            regionserver.BloomType
            ClusterStatus
            HBaseConfiguration HColumnDescriptor HTableDescriptor
            HRegionInfo
            RegionLoad
            TableName
            client.Durability
            client.Connection
            client.Admin]
           org.slf4j.Logger
           org.slf4j.LoggerFactory))

(def ^:private ^Logger logger
  (LoggerFactory/getLogger (str *ns*)))

(defmacro ^:private info [& args]
  `(.info logger ~@args))

(def ^:private enum->keyword (comp keyword #(str/replace % "_" "-") str/lower-case str))

(def ^:private alter-monitor-interval-ms 2000)

(defn- keyword->enum-map [enums]
  (into {} (for [e enums] [(enum->keyword e) (str e)])))

;;; FIXME - Unreliable
(defn- constants-of
  "Returns the list of constant names of the class"
  [class]
  (->> (refl/reflect class)
       :members
       (filter (comp #(every? % [:public :static]) :flags))
       (remove :return-type)
       (map :name)
       (map str)
       (remove #(str/starts-with? % "IS_"))
       (map #(str/replace % #"^DEFAULT_" ""))
       set))

(defn- constant-of
  "Returns the upper-case string of the constant name if the given class has it"
  [class key]
  (let [upper-name (str/replace (str/upper-case (name key)) "-" "_")
        constants  (constants-of class)]
    (get constants upper-name)))

(def ^:private valid-types
  {:bloomfilter         (keyword->enum-map (BloomType/values))
   :compression         (keyword->enum-map (Compression$Algorithm/values))
   :data-block-encoding (keyword->enum-map (DataBlockEncoding/values))
   :durability          (keyword->enum-map (Durability/values))})

(def supported-properties
  (letfn [(as-keywords [klass]
            (conj (map #(-> %
                            str/lower-case
                            (str/replace "_" "-")
                            keyword)
                       (constants-of klass))
                  :configuration))]
    {:table  (as-keywords HTableDescriptor)
     :column (as-keywords HColumnDescriptor)
     :valid-options valid-types}))

(defn- value-for
  [key val]
  (if-let [k->e (get valid-types key)]
    (if-let [val (k->e val)]
      val
      (throw (IllegalArgumentException.
               (format "Invalid value for %s: %s" key val))))
    (str val)))

(defn- java-new [java-class & args]
  (clojure.lang.Reflector/invokeConstructor
    java-class
    (into-array Object args)))

(defn- patch-descriptor
  [desc spec]
  (doseq [[key val] (dissoc spec :splits)]
    (case key
      :configuration
      (doseq [[ckey cval] val]
        (if (nil? cval)
          (.removeConfiguration desc ckey)
          (.setConfiguration desc ckey (str cval))))
      (if-let [const (constant-of (class desc) key)]
        (if (nil? val)
          (.remove desc (Bytes/toBytes const))
          (.setValue desc (Bytes/toBytes const)
                     (Bytes/toBytes (value-for key val))))
        (if (string? key)
          (if (nil? val)
            (.remove desc key)
            (.setValue desc key (str val)))
          (throw (IllegalArgumentException.
                   (str "Unknown property: " key)))))))
  desc)

(defn spec->desc
  [java-class arg spec]
  (patch-descriptor (java-new java-class arg) spec))

(defn ^Admin get-admin
  [^Connection conn]
  (.getAdmin conn))

(defmacro with-admin
  [[name obj] & forms]
  (let [name (vary-meta name assoc :tag `Admin)]
    `(if (instance? Admin ~obj)
       (let [~name ~obj] ~@forms)
       (with-open [~name (get-admin ~obj)] ~@forms))))

(defn tables
  "Returns table descriptors indexed by their names"
  [conn]
  (with-admin [admin conn]
    (into {}
          (map #(vector (.. ^HTableDescriptor % getTableName getNameAsString) %)
               (.listTables admin)))))

(defn exists?
  "Checks if the table exists"
  [conn table-name]
  (with-admin [admin conn]
    (.tableExists admin (table-name-for table-name))))

;;; https://issues.apache.org/jira/browse/HBASE-11862
;;; - Writables will be replaced to Bytes, but not yet in 1.2 branch
#_(defn coprocessors
  "Returns the coprocessors for the table with index numbers"
  [conn table-name]
  (with-admin [admin conn]
    (let [desc (describe-table conn table-name)]
      (for [[^Bytes k ^Bytes v] (.getValues desc)]
        (bytes-> :string (.get k))))))

(defn ^HTableDescriptor describe-table
  "Returns table descriptor of the table"
  [conn table-name]
  (with-admin [admin conn]
    (.getTableDescriptor admin (table-name-for table-name))))

#_(def create-table nil)
(defmulti create-table
  "([conn table-name spec & [table-props]])
  spec => HTableDescriptor | Map | Keyword

  Creates an HBase table. spec can either be the name of the column family, an
  HTableDescriptor, or a map describing the properties of the column families."
  (fn [conn table-name spec & [table-props]] (class spec)))

(defmethod create-table HTableDescriptor
  [conn table-name table-descriptor & [table-props]]
  (let [table-name (table-name-for table-name)
        splits (some->> (:splits table-props)
                        (map ->bytes)
                        into-array)
        ;; Based on the original table descriptor
        new-descriptor ^HTableDescriptor (spec->desc HTableDescriptor
                                                     table-descriptor
                                                     table-props)]
    ;; FIXME - setName method is deprecated
    (.setName new-descriptor table-name)
    (with-admin [admin conn]
      (if (seq splits)
        (.createTable admin new-descriptor splits)
        (.createTable admin new-descriptor)))))

(defmethod create-table clojure.lang.IPersistentMap
  [conn table-name column-families & [table-props]]
  (let [^HTableDescriptor htd
        (spec->desc HTableDescriptor (table-name-for table-name) {})]
    (doseq [[cf-name spec] column-families]
      (.addFamily htd (spec->desc HColumnDescriptor (name cf-name) spec)))
    (create-table conn table-name htd table-props)))

(defmethod create-table clojure.lang.Keyword
  [conn table-name cf-name & [table-props]]
  (create-table conn table-name {cf-name nil} table-props))

(defn- wait-alter-completion
  "Waits until alteration is complete"
  [^Admin admin table-name]
  (loop [prev 0]
    (let [pair (.getAlterStatus admin ^TableName table-name)
          yet  (int (.getFirst pair))
          all  (int (.getSecond pair))]
      (if (not= yet prev)
        (info (format "Altering %s (%d/%d)"
                          (str table-name) (- all yet) all)))
      (when-not (zero? yet)
        (Thread/sleep alter-monitor-interval-ms)
        (recur yet)))))

(defmacro ^:private with-alter
  [[admin table-name message wait] & forms]
  `(do
     (info ~message)
     ~@(map
         (fn [f]
           `(~(first f) ~admin ~table-name ~@(rest f)))
         forms)
     (if ~wait (wait-alter-completion ~admin ~table-name))))

(defn- alter-table-
  [wait conn table-name families & [table-props]]
  (with-admin [admin conn]
    (let [table-name (table-name-for table-name)
          htd        (describe-table admin table-name)]
      (doseq [[cf-name spec] families]
        (let [cf-name-str (name cf-name)
              cf-name     (->bytes cf-name-str)]
          (cond
            (nil? spec)
            (if (.hasFamily htd cf-name)
              (with-alter [admin table-name (str "Drop CF: " cf-name-str) wait]
                (.deleteColumn cf-name)))

            (map? spec) ; TODO HColumnDescriptor
            (if-let [hcd (.getFamily htd cf-name)]
              (with-alter [admin table-name (str "Alter CF: " cf-name-str) wait]
                (.modifyColumn (patch-descriptor hcd spec)))
              (with-alter [admin table-name (str "Add CF: " cf-name-str) wait]
                (.addColumn (spec->desc HColumnDescriptor cf-name spec))))

            :else
            (throw (IllegalArgumentException.
                     (str "Invalid alter request: " spec))))))

      (if (seq table-props)
        (with-alter [admin table-name (str "Alter table: " table-name) wait]
          ;; Retrieve updated HTableDescriptor
          (.modifyTable (patch-descriptor (describe-table admin table-name) table-props))))

      (info (str "Alter table complete: " table-name)))))

(defn alter-table
  "Alters the table and waits for completion"
  [conn table-name families & [table-props]]
  (alter-table- true conn table-name families table-props))

(defn alter-table!
  "Alters the table without waiting"
  [conn table-name families & [table-props]]
  (alter-table- false conn table-name families table-props))

(defmacro ^:private deftablefn
  [name docstring & methods]
  `(defn ~name
     ~docstring
     [~'conn ~'table-name]
     (with-admin
       [~'admin ~'conn]
       (let [~'table-name (table-name-for ~'table-name)]
         ~@(map #(list % 'admin 'table-name)
                methods)))))

(deftablefn enable-table  "Enables the table" .enableTable)
(deftablefn disable-table "Disables the table" .disableTable)
(deftablefn enabled? "Checks if the table is enabled" .isTableEnabled)

(defn drop-table
  "Drops the table"
  [conn table-name]
  (with-admin [admin conn]
    (let [table-name (table-name-for table-name)]
      (if (.isTableEnabled admin table-name)
        (.disableTable admin table-name))
      (.deleteTable admin table-name))))

(defmacro ^:private defbang [& funcs]
  (map #(list 'defn (symbol (str % "!")) '[& args]
              (list 'try
                    (list 'apply % 'args)
                    '(catch Exception e)))
       funcs))

(defbang drop-table enable-table disable-table)

(defn name-comparator
  "Comparator function for server names. Pads numbers with zeros"
  [left right]
  (let [fmt   (fn [s] (str/replace s #"[0-9]+" #(format "%04d" (Long/parseLong %))))
        left  (fmt left)
        right (fmt right)]
    (.compareTo ^String left right)))

(defn- calculate-locality
  [regions]
  (let [[loc-size tot-size] (reduce #(let [{size :size-mb loc :locality
                                            :or {size 0 loc 0}} (val %2)]
                                       (map + %1 [(* size loc) size]))
                                    [0 0] regions)]
    (if (zero? tot-size) 0 (/ loc-size tot-size))))

(defn- aggregate-region-info
  [server-name infos loads has-locality]
  (let [from-infos (into {}
                         (for [^HRegionInfo info infos]
                           [(.getEncodedName info)
                            {:name         (.getRegionName info)
                             :encoded-name (.getEncodedName info)
                             :server       (str server-name)
                             :start-key    (.getStartKey info)
                             :end-key      (.getEndKey info)
                             :table        (str (.getTable info))
                             :online?      (not (.isOffline info))}]))
        from-loads (into {}
                         (for [[name-bytes ^RegionLoad load] loads]
                           [(HRegionInfo/encodeRegionName name-bytes)
                            (conj
                              {:memstore-mb          (.getMemStoreSizeMB load)
                               :size-mb              (.getStorefileSizeMB load)
                               :storefiles           (.getStorefiles load)
                               :uncompressed-size-mb (.getStoreUncompressedSizeMB load)}
                              (if has-locality
                                {:locality (.getDataLocality load)}))]))]
    (merge-with merge from-infos from-loads)))

(defn- expand-server-load
  "Collects information of server and its regions and returns it as a map"
  [^Admin admin ^ClusterStatus status server-name]
  (let [load (.getLoad status server-name)
        has-locality (not= \0 (first (.getHBaseVersion status))) ;; TODO
        report       {:storefiles   (.getStorefiles load)
                      :max-heap-mb  (.getMaxHeapMB load)
                      :memstore-mb  (.getMemstoreSizeInMB load)
                      :used-heap-mb (.getUsedHeapMB load)
                      :reads        (.getReadRequestsCount load)
                      :writes       (.getWriteRequestsCount load)
                      :regions      (aggregate-region-info
                                      server-name
                                      (.getOnlineRegions admin server-name)
                                      (.getRegionsLoad load)
                                      has-locality)}]
    (assoc report :locality (calculate-locality (:regions report)))))

(defn cluster-status
  "Returns the cluster status as a map"
  [conn]
  (with-admin [admin conn]
    (let [status (.getClusterStatus admin)
          report {:avg-load       (.getAverageLoad status)
                  :master         (str (.getMaster status))
                  :backup-masters (sort name-comparator
                                        (map str (.getBackupMasters status)))
                  :servers        (.getServers status)
                  :balancer-on?   (.isBalancerOn status)
                  :requests       (.getRequestsCount status)
                  :hbase-version  (.getHBaseVersion status)}
          report (update report :servers
                         #(into (sorted-map-by name-comparator)
                                (for [s %] [(str s) (expand-server-load admin status s)])))]
      report)))

(defn servers
  "Returns the list of regionservers"
  [conn]
  (with-admin [admin conn]
    (let [status  (.getClusterStatus admin)
          servers (.getServers status)]
      (vec (map str servers)))))

(defn- sort-regions-fn
  [regions]
  (fn [& args]
    (let [props     (map regions args)
          table-cmp (apply compare (map :table props))]
      (if-not (zero? table-cmp)
        table-cmp
        (let [[left right] (map (comp #(or % empty-bytes)
                                      :start-key) props)]
          (Bytes/compareTo left right))))))

(defn regions
  "Returns the information of online regions as a sorted map. Regions are
  sorted first by their table names then by start keys"
  [conn & tables]
  (let [filt (if (seq tables)
               (comp (set (map name tables)) :table val)
               identity)]
    (with-admin [admin conn]
      (let [unsorted
            (->> (cluster-status conn)
                 :servers
                 (map val)
                 (map :regions)
                 (reduce merge)
                 (filter filt)
                 (into {}))]
        (into (sorted-map-by (sort-regions-fn unsorted))
              unsorted)))))

