(ns hbase-clj.core
  (:refer-clojure :exclude [get inc])
  (:require [clojure.core :as c]
            [hbase-clj.admin :as admin]
            [hbase-clj.ops   :as ops]
            [hbase-clj.util  :refer [bytes-> table-name-for]])
  (:import [org.apache.hadoop.hbase.client
            Connection ConnectionFactory Table
            BufferedMutator BufferedMutatorParams
            BufferedMutator$ExceptionListener
            Result ResultScanner]
           org.apache.hadoop.conf.Configuration
           org.apache.hadoop.hbase.HBaseConfiguration
           org.apache.hadoop.security.UserGroupInformation
           java.net.URI
           javax.security.auth.login.AppConfigurationEntry
           com.sun.security.auth.login.ConfigFile
           org.slf4j.Logger
           org.slf4j.LoggerFactory))

(def ^:private ^Logger logger
  (LoggerFactory/getLogger (str *ns*)))

(defmacro ^:private info [& args]
  `(.info logger ~@args))

;;; Connecting to HBase
(defn- ^Configuration build-configuration
  "Creates HBaseConfiguration object according to the given map
  If no argument is given, the configuration will be built with HBase
  resources."
  ([]
   (HBaseConfiguration/create))
  ([arg]
   (let [conf (build-configuration)]
     (doseq [[k v] arg]
       (.set conf k (str v)))
     conf)))

(defn- find-context [jaas]
  (let [content (slurp jaas)]
    (->> (re-seq
           #"(?si)(\S+)\s*\{.*?Krb5LoginModule\s*required(.*?)\}"
           content)
         (map #(nth % 1))
         first)))

(defn ^Connection connect
  "[props] => String | Map | org.apache.hadoop.conf.Configuration

  Connects to HBase. The returned connection object is AutoCloseable so it can
  be used with with-open macro.

  The argument can be either a string which is interpreted as the value of
  `hbase.zookeeper.quorum' or a map of configuration parameters.

      (connect \"zk1,zk2,zk3\")
      (connect {\"hbase.zookeeper.quorum\" \"zk1,zk2,zk3\"
                \"hbase.hconnection.threads.core\" 128})

  When connecting to a secured HBase cluster with Kerberos authentication, the
  map must contain the path to JAAS login configuration file as :jaas entry.

      (System/setProperty \"java.security.krb5.conf\" \"/etc/krb5.conf\")
      (def C
        (connect {:jaas \"/tmp/jaas.conf\"
                  \"hbase.zookeeper.quorum\" \"zk1,zk2,zk3\"
                  \"hadoop.security.authentication\" \"kerberos\"
                  \"hbase.security.authentication\" \"kerberos\"
                  \"hbase.master.kerberos.principal\" \"hbase/_HOST@EXAMPLE.COM\"
                  \"hbase.regionserver.kerberos.principal\" \"hbase/_HOST@EXAMPLE.COM\"}))

  If the file contains multiple login contexts, the first one will be chosen,
  but you can explicitly specify the name of login context as :context entry.
  And if the file does not contain keyTab entry, you have to pass the full path
  to the keytab file as :keytab."
  ([]
   (connect (build-configuration)))
  ([props]
   (cond
     (map? props)
     (let [{:keys [context jaas keytab pool]} props
           conf    (build-configuration (dissoc props :context :jaas :pool :keytab))]
       (when jaas
         (let [context (or context (find-context jaas))]
           (info (str "Login context: " context)) ; FIXME
           (javax.security.auth.login.Configuration/setConfiguration
             (ConfigFile. (URI. (str "file://" jaas))))
           (if-let [login (some->
                            (javax.security.auth.login.Configuration/getConfiguration)
                            (.getAppConfigurationEntry context)
                            first
                            .getOptions)]
             (let [{:strs [principal keyTab]} login
                   keyTab (or keyTab keytab)]
               (if-not keyTab
                 (throw (IllegalArgumentException.
                          (str "No 'keyTab' in login context: " context))))
               (sun.security.krb5.Config/refresh)
               (UserGroupInformation/setConfiguration conf)
               (UserGroupInformation/loginUserFromKeytab principal keyTab)
               (info "Current user: " (UserGroupInformation/getCurrentUser))) ; FIXME
             (throw (IllegalArgumentException.
                      (str "Invalid login context: " context ""))))))
       (ConnectionFactory/createConnection conf pool))

     (instance? org.apache.hadoop.conf.Configuration props)
     (ConnectionFactory/createConnection props)

     :else (connect {"hbase.zookeeper.quorum" props}))))

(defn ^Table get-table
  [^Connection conn name]
  (.getTable conn (table-name-for name)))

(defmacro with-table
  "Executes the forms with the bindings to Table instances that are
  automatically closed. The bound instances are not thread-safe."
  [conn bindings & forms]
  `(with-open
     ~(reduce into []
              (for [[bind table-name] (partition 2 bindings)]
                [(vary-meta bind assoc :tag `Table)
                 `(get-table ~conn ~table-name)]))
     ~@forms))

(defmacro with-buffered-mutator
  "Executes the forms with the bindings to BufferedMutator instances that are
  automatically closed. The bound instances are not thread-safe.

  Accepts the following options:
  :on-exception - Function to call on exception. Parameters: [e bm]
  :pool         - Optional thread pool"
  [conn bindings-and-options & forms]
  (let [{bindings false
         options  true} (group-by (comp keyword? first)
                                  (partition 2 bindings-and-options))
        options (into {} (map vec options))
        {:keys [on-exception pool]} options]
    `(with-open
       ~(reduce into []
                (for [[bind table-name] bindings]
                  [(vary-meta bind assoc :tag `BufferedMutator)
                   `(.getBufferedMutator
                      ~(vary-meta conn assoc :tag `Connection)
                      (let [bmp# (BufferedMutatorParams. (table-name-for ~table-name))
                            xfn# ~on-exception]
                        (if xfn#
                          (.listener bmp# (reify
                                            BufferedMutator$ExceptionListener
                                            (onException [this# e# bm#]
                                              (xfn# e# bm#)))))
                        (if ~pool
                          (.pool bmp# ~pool))
                        bmp#))]))
       ~@forms)))

;;; Basic HBase operations
;;; - put
;;; - inc
;;; - del
;;; - append
;;;
;;; - get
;;; - scan
;;;
;;; - batch
;;;
;;; Reference:
;;;   https://hbase.apache.org/apidocs/org/apache/hadoop/hbase/client/Table.html

(defn batch
  "Executes multiple operations in a batch"
  [^Table table ops]
  (let [results (make-array Object (count ops))]
    (.batch table ops results)
    (vec results)))

(defn put
  "data => rowkey {key val ...}...

  Puts the data to the table"
  [^Table table & data]
  (.put table (map (partial apply ops/put)
                   (partition 2 data))))

(defn inc
  "inc-map => {col inc-by ...}

  Increments column values."
  [^Table table rowkey inc-map]
  (.increment table (ops/inc rowkey inc-map)))

(defn append
  "append-map => {col data ...}

  Appends data to the column"
  [^Table table rowkey append-map]
  (.append table (ops/append rowkey append-map)))

(defn del
  "Deletes the record from the table"
  [^Table table rowkey]
  (.delete table (ops/del rowkey)))

(defn get
  "Retrieves a single row from the table if a single rowkey is specified or
  multiple rows at one if multiple rowkeys are given as a collection

  Accepts the following options:
    :filters      - List of filters
    :cache-blocks - Whether blocks should be cached
    :versions     - Number of versions to fetch
    :fn           - Function that manipulates the Get object is construction"
  [^Table table & args]
  (.get table (apply ops/get args)))

(defn ^ResultScanner scan
  "Scans the table with the given options.

  Accepts the following options:
    :range        - Rowkey range: [start-key stop-key]
    :time-range   - Timestamp range: [start-ts end-ts]
    :project      - Columns (or column families) to fetch
    :filters      - List of filters
    :while        - Additional filters that will cause early termination of scan
    :small        - Whether this scan is a small one (default: false)
    :reversed     - Whether this scan is a reversed one (default: false)
    :raw          - Whether to include delete markers and deleted rows
    :batch        - Number of values to fetch for a single call to next()
    :caching      - Number of rows to cache
    :cache-blocks - Whether blocks should be cached
    :max-size     - Maximum size of the result
    :versions     - Number of versions to fetch
    :replica-id   - Region replica id where scan will run
    :id           - Identifier for the scan
    :fn           - Function that manipulates the Scan object in construction"
  [table & [{:as options}]]
  (.getScanner table (ops/scan options)))

