(ns hbase-clj.filter
  (:refer-clojure :exclude [= > >= < <= and or while])
  (:require [hbase-clj.util :refer [parse-column ->bytes]])
  (:import [org.apache.hadoop.hbase.filter
            ByteArrayComparable
            FilterList
            FilterList$Operator
            FirstKeyOnlyFilter
            FuzzyRowFilter
            KeyOnlyFilter
            CompareFilter$CompareOp
            SingleColumnValueFilter
            BinaryComparator
            RegexStringComparator
            PrefixFilter
            SkipFilter
            RowFilter
            WhileMatchFilter]
           org.apache.hadoop.hbase.util.Pair))

;;; https://hbase.apache.org/apidocs/org/apache/hadoop/hbase/filter/package-summary.html

(defn and
  "Returns conjunctive FilterList"
  [& filters]
  (if (seq (rest filters))
    (FilterList. filters)
    (first filters)))

(defn or
  "Returns disjunctive FilterList"
  [& filters]
  (if (seq (rest filters))
    (FilterList. FilterList$Operator/MUST_PASS_ONE filters)
    (first filters)))

(defn while
  [& filters]
  (if (seq filters)
    (WhileMatchFilter. (apply and filters))
    nil))

(defn bin-comparable
  [val]
  (if (instance? ByteArrayComparable val)
    val
    (BinaryComparator. (->bytes val))))

(defn- value
  "Returns SingleColumnValueFilter for the operation"
  [cfcq op val]
  (let [[cf cq] (parse-column cfcq)]
    (if (nil? cq)
      (throw (IllegalArgumentException. "Qualifier must be given")))
    (doto (SingleColumnValueFilter.
            cf cq op (bin-comparable val))
      (.setFilterIfMissing true))))

(defn =
  "Returns filter that matches rows with the column whose value matches the
  given value. The value can be either a byte array or a regular expression."
  [cfcq val]
  (value cfcq CompareFilter$CompareOp/EQUAL
         (if (instance? java.util.regex.Pattern val)
           (RegexStringComparator. (str val))
           val)))

(defn >
  "Returns column filter with greater-than condition"
  ([cfcq val]
   (value cfcq CompareFilter$CompareOp/GREATER val))
  ([max cfcq min]
   (and
     (value cfcq CompareFilter$CompareOp/LESS max)
     (value cfcq CompareFilter$CompareOp/GREATER min))))

(defn >=
  "Returns column filter with greater-than-or-equals-to condition"
  ([cfcq val]
   (value cfcq CompareFilter$CompareOp/GREATER_OR_EQUAL val))
  ([max cfcq min]
   (and
     (value cfcq CompareFilter$CompareOp/LESS_OR_EQUAL max)
     (value cfcq CompareFilter$CompareOp/GREATER_OR_EQUAL min))))

(defn <
  "Returns column filter with less-than condition"
  ([cfcq val]
   (value cfcq CompareFilter$CompareOp/LESS val))
  ([min cfcq max]
   (and
     (value cfcq CompareFilter$CompareOp/GREATER min)
     (value cfcq CompareFilter$CompareOp/LESS max))))

(defn <=
  "Returns column filter with less-than-or-equals-to condition"
  ([cfcq val]
   (value cfcq CompareFilter$CompareOp/LESS_OR_EQUAL val))
  ([min cfcq max]
   (and
     (value cfcq CompareFilter$CompareOp/GREATER_OR_EQUAL min)
     (value cfcq CompareFilter$CompareOp/LESS_OR_EQUAL max))))

(defn prefix
  "Returns filter that selects rows that match the given prefix"
  [p]
  (PrefixFilter. (->bytes p)))

(defn row=
  "Returns filter that selects rows whose rowkeys match the given value or
  regular expression"
  [val]
  (RowFilter.
    CompareFilter$CompareOp/EQUAL
    (if (instance? java.util.regex.Pattern val)
      (RegexStringComparator. (str val))
      (bin-comparable val))))

(defn row>
  "Returns filter that selects rows whose rowkeys are greater than the given
  value"
  [val]
  (RowFilter. CompareFilter$CompareOp/GREATER (bin-comparable val)))

(defn row>=
  "Returns filter that selects rows whose rowkeys are greater than or equal to
  the given value"
  [val]
  (RowFilter. CompareFilter$CompareOp/GREATER_OR_EQUAL (bin-comparable val)))

(defn row<
  "Returns filter that selects rows whose rowkeys are less than the given
  value"
  [val]
  (RowFilter. CompareFilter$CompareOp/LESS (bin-comparable val)))

(defn row<=
  "Returns filter that selects rows whose rowkeys are less than or equal to the
  given value"
  [val]
  (RowFilter. CompareFilter$CompareOp/LESS_OR_EQUAL (bin-comparable val)))

(defn fuzzy-row
  "Returns fuzzy row filter"
  [fuzzy-key fuzzy-mask]
  (FuzzyRowFilter. [(Pair. fuzzy-key fuzzy-mask)]))

(defn first-cell-only
  "Returns FirstKeyOnlyFilter"
  []
  (FirstKeyOnlyFilter.))

(defn key-only
  "Returns KeyOnlyFilter"
  []
  (KeyOnlyFilter.))

(defn skip
  "Returns a filter that wraps the given filter and skips the entire row if any
  of the cell checks passes"
  [f]
  (SkipFilter. f))
