(ns hbase-clj.ops
  "Namespace for functions for building objects that represent different types
  of HBase operations."
  (:refer-clojure :exclude [get inc])
  (:require [hbase-clj.filter :as f]
            [hbase-clj.util :refer [->bytes parse-column]])
  (:import [org.apache.hadoop.hbase.client
            Get Scan Put Delete Increment Append]
           [org.apache.hadoop.hbase.filter Filter FilterList]))

(defn ^Get get
  "Builds a Get object or a list of Get objects for the arguments.

  Accepts the following options:
    :filters      - List of filters
    :cache-blocks - Whether blocks should be cached
    :versions     - Number of versions to fetch
    :fn           - Function that manipulates the Get object is construction"
  [rowkeys & [{:keys [filters cache-blocks versions fn]
               :or {filters      []
                    cache-blocks nil
                    versions     nil
                    fn           identity}
               :as options}]]
  {:pre [(some #(% options) [map? nil?])]}
  (letfn [(proc [^Get g]
            (case (count filters)
              0 nil
              1 (.setFilter g ^Filter (first filters))
              (.setFilter g (FilterList. filters)))

            (if versions
              (.setMaxVersions g versions))

            (if (some? cache-blocks)
              (.setCacheBlocks g cache-blocks))

            (fn g))]
    (if (coll? rowkeys)
      (for [rowkey rowkeys]
        (proc (Get. ^bytes (->bytes rowkey))))
      (proc (Get. ^bytes (->bytes rowkeys))))))

(defn ^Scan scan
  "Builds Scan object for the arguments.

  Accepts the following options:
    :range        - Rowkey range: [start-key stop-key]
    :time-range   - Timestamp range: [start-ts end-ts]
    :project      - Columns (or column families) to fetch
    :filters      - List of filters
    :while        - Additional filters that will cause early termination of scan
    :small        - Whether this scan is a small one (default: false)
    :reversed     - Whether this scan is a reversed one (default: false)
    :raw          - Whether to include delete markers and deleted rows
    :batch        - Number of values to fetch for a single call to next()
    :caching      - Number of rows to cache
    :max-size     - Maximum size of the result
    :cache-blocks - Whether blocks should be cached
    :versions     - Number of versions to fetch
    :replica-id   - Region replica id where scan will run
    :id           - Identifier for the scan
    :fn           - Function that manipulates the Scan object in construction"
  [& [{:keys [range time-range project filters while small reversed
              raw batch caching cache-blocks max-size versions replica-id id fn]
       :or {range        []
            time-range   []
            project      []
            filters      []
            while        []
            small        false
            reversed     false
            raw          false
            batch        nil
            caching      nil
            cache-blocks nil
            max-size     nil
            versions     nil
            replica-id   nil
            id           nil
            fn           identity}
       :as options}]]
  {:pre [(some #(% options) [map? nil?])]}
  (let [scan    (Scan.)
        range   (map ->bytes range)
        filters (if (seq while)
                  (conj filters (apply f/while while))
                  filters)]

    ;; Apply rowkey range
    (case (count range)
      0 nil
      1 (.setStartRow scan (first range))
      2 (let [[start stop] range]
          (when start (.setStartRow scan start))
          (when stop  (.setStopRow  scan stop)))
      (throw (IllegalArgumentException. ":range expects 1..2 element(s)")))

    ;; Apply timestamp range
    (case (count time-range)
      0 nil
      1 (.setTimeStamp scan (first time-range))
      2 (.setTimeRange scan
                       (or (first time-range) 0)
                       (or (last  time-range) Long/MAX_VALUE))
      (throw (IllegalArgumentException. ":time-range expects at most two elements")))

    ;; Apply projection
    (doseq [cfcq project]
      (let [[cf cq] (parse-column cfcq)]
        (if (nil? cq)
          (.addFamily scan cf)
          (.addColumn scan cf cq))))

    ;; Apply filters
    (case (count filters)
      0 nil
      1 (.setFilter scan ^Filter (first filters))
      (.setFilter scan (FilterList. filters)))

    ;; Small scan
    (if small (.setSmall scan true))

    ;; Reversed
    (if reversed (.setReversed scan true))

    ;; Raw
    (if raw (.setRaw scan true))

    ;; Batch
    (if batch (.setBatch scan batch))

    ;; Caching
    (if caching (.setCaching scan caching))

    ;; Max result size
    (if max-size (.setMaxResultSize scan max-size))

    ;; Block caching
    (if (some? cache-blocks)
      (.setCacheBlocks scan cache-blocks))

    ;; Versions
    (if versions (.setMaxVersions scan versions))

    ;; Replica-id
    (if replica-id (.setReplicaId scan ^int replica-id))

    ;; Id
    (if id (.setId scan ^String id))

    ;; Apply transformer function
    (fn scan)

    scan))

(defn ^Put put
  "data => {key val ...}

  Builds a Put object for the given arguments."
  [rowkey data]
  (let [put (Put. ^bytes (->bytes rowkey))]
    (doseq [[cfcq val] data]
      (let [[^bytes cf ^bytes cq] (parse-column cfcq)]
        (if (map? val)
          (doseq [[^long ts v] val]
            (.addColumn put cf cq ts ^bytes (->bytes v)))
          (.addColumn put cf cq (->bytes val)))))
    put))

(defn ^Increment inc
  "inc-map => {col inc-by ...}

  Increments one or more columns within a single row"
  [rowkey inc-map]
  (let [i (Increment. ^bytes (->bytes rowkey))]
    (doseq [[k by] inc-map]
      (let [[cf cq] (parse-column k)]
        (.addColumn i cf cq by)))
    i))

(defn ^Append append
  "append-map => {col data ...}

  Appends data to the column"
  [rowkey append-map]
  (let [a (Append. ^bytes (->bytes rowkey))]
    (doseq [[k tail] append-map]
      (let [[cf cq] (parse-column k)]
        (.add a cf cq (->bytes tail))))
    a))

(defn ^Delete del
  "TODO Delete granularity"
  [rowkey]
  (let [d (Delete. ^bytes (->bytes rowkey))]
    d))

