(ns hbase-clj.result
  (:require [hbase-clj.util :refer [bytes->]])
  (:import [org.apache.hadoop.hbase.client Result Put]))

(defn record-size
  "Returns total size of raw cells"
  [^Result result]
  (Result/getTotalSizeOfCells result))

(defn partial?
  "Checks if the result is partial"
  [^Result result]
  (.isPartial result))

(defn stale?
  "Checks if the result is stale"
  [^Result result]
  (.isStale result))

(defn rowkey
  "Returns the row key in bytes"
  ([^Result result]
   (.getRow result))
  ([type ^Result result]
   (bytes-> type (.getRow result))))

(defn- lookup-schema
  [schema cf cq]
  (if (keyword? schema)
    schema
    (if-let [t (or (get-in schema [cf cq])
                   (schema (keyword (str (name cf) ":" (name cq)))))]
      t
      (as-> (get schema cf) $
        (filter #(and
                   (instance? java.util.regex.Pattern (key %))
                   (re-seq (key %) (str cq))) $)
        (map val $)
        (first $)
        (or $ :bytes)))))

;;; TODO
;;; - binary qualifiers
(defn- deep-merge
  "Merge maps recursively"
  [& maps]
  (reduce #(if (every? map? [%1 %2])
             (merge-with deep-merge %1 %2)
             %2) (empty (first maps)) maps))

(defn ->map
  "schema    => type
               | {type-pair ...}
  type-pair => {cf:cq type}
               | {cf {cq type ...}}
  cf:cq     => <KEYWORD>
  cf        => <KEYWORD>
  cq        => <KEYWORD>
               | <REGEXP>
  type      => :string
               | :string-binary
               | :keyword
               | :float
               | :double
               | :int
               | :short
               | :long
               | :bigdecimal

  Interpretes a Result object with the given schema and returns the map
  representation of it. CFs and CQs are returned in keywords which means it is
  not designed to be used with the result from flat-wide tables.

      {:cf:cq val ...}"
  ([^Result result]
   (->map {} result))
  ([schema ^Result result]
   (if-not (.isEmpty result)
     (let [raw-map (.getNoVersionMap result)
           rowkey  (.getRow result)
           pairs (for [[cf cq-val] raw-map
                       [cq val]    cq-val]
                   (let [cf   (bytes-> :string cf)
                         cq   (bytes-> :string cq)
                         cfcq (keyword (str cf ":" cq))
                         t    (lookup-schema schema cf cq)]
                     [cfcq (bytes-> t val)]))]
       (with-meta (into (sorted-map) pairs) {:rowkey rowkey})))))

(defn ->all-map
  "Similar to ->map, but returns all versions indexed by their timestamps.

      {:cf:cq {ts val ...} ... ...}"
  ([^Result result]
   (->all-map {} result))
  ([schema ^Result result]
   (if-not (.isEmpty result)
     (let [raw-map (.getMap result)
           rowkey (.getRow result)
           spread (for [[cf cq-ts-val] raw-map
                        [cq ts-val] cq-ts-val
                        [ts val] ts-val]
                    (let [cf   (bytes-> :string cf)
                          cq   (bytes-> :string cq)
                          cfcq (keyword (str cf ":" cq))
                          t    (lookup-schema schema cf cq)]
                      {cfcq (sorted-map-by > ts (bytes-> t val))}))]
       (with-meta (apply deep-merge spread) {:rowkey rowkey})))))

(defn ->put
  "Converts Result object into Put object"
  [^Result result]
  (let [put (Put. (.getRow result))]
    (doseq [cell (.rawCells result)]
      (.add put cell))
    put))
