(ns hbase-clj.util
  "Utility functions"
  (:require [clojure.string :as str]
            [clojure.pprint :refer [pprint]])
  (:import [org.apache.hadoop.hbase.util
            Bytes OrderedBytes]
           org.apache.hadoop.hbase.TableName))

(def empty-bytes (byte-array 0))
(def ^:private byte-array-class (class empty-bytes))

(def bytes-comparator
  "Comparator for byte arrays"
  Bytes/BYTES_COMPARATOR)

(defmulti  ^:private   type->bytes          class)
(defmethod type->bytes nil                  [_] empty-bytes)
(defmethod type->bytes byte-array-class     [e] e)
(defmethod type->bytes clojure.lang.Keyword [e] (Bytes/toBytes (name                e)))
(defmethod type->bytes Byte                 [e] (byte-array    1                    e))
(defmethod type->bytes Short                [e] (Bytes/toBytes ^short               e))
(defmethod type->bytes Integer              [e] (Bytes/toBytes ^int                 e))
(defmethod type->bytes Long                 [e] (Bytes/toBytes ^long                e))
(defmethod type->bytes Float                [e] (Bytes/toBytes ^float               e))
(defmethod type->bytes Double               [e] (Bytes/toBytes ^double              e))
(defmethod type->bytes String               [e] (Bytes/toBytes ^String              e))
(defmethod type->bytes java.nio.ByteBuffer  [e] (Bytes/toBytes ^java.nio.ByteBuffer e))
(defmethod type->bytes BigDecimal           [e] (Bytes/toBytes ^BigDecimal          e))
(defmethod type->bytes Boolean              [e] (Bytes/toBytes ^Boolean             e))
(defmethod type->bytes :default             [e] (Bytes/toBytes e))

(defn ->bytes
  "Converts arguments to java byte arrays and concatenates the arrays.

    e.g. (->bytes \"hello\" 1 (int 2) (short 3) (byte 4))"
  [& args]
  (if (= 1 (count args))
    (type->bytes (first args))
    (->> args
         (mapcat type->bytes)
         byte-array)))

(defn bytes->
  "Converts byte array to the given type.

  Supported types are:
    :string
    :string-binary
    :keyword
    :float
    :double
    :byte
    :short
    :int
    :long
    :bigdecimal
    :bytes"
  [t ^bytes val]
  (case t
    :string        (Bytes/toString val)
    :string-binary (Bytes/toStringBinary val)
    :keyword       (keyword (Bytes/toString val))
    :float         (Bytes/toFloat val)
    :double        (Bytes/toDouble val)
    :int           (Bytes/toInt val)
    :short         (Bytes/toShort val)
    :long          (Bytes/toLong val)
    :bigdecimal    (Bytes/toBigDecimal val)
    :byte          (first val)
    :bytes val
    nil val
    (throw (IllegalArgumentException. (str "Unknown type: " t)))))

;;;
;;; Handling column key expressions
;;;
(defmulti parse-column
  "Multimethod that parses column key (CF:CQ) expression.
  If the argument is given as a collection of size 2, it is interpreted as
  a pair of CF and CQ."
  (fn [cfcq]
    (if (coll? cfcq)
      :collection
      (class cfcq))))

(defmethod parse-column :collection
  [cfcq]
  (if-not (= (count cfcq) 2)
    (throw (IllegalArgumentException. "Invalid column expression")))
  (map ->bytes cfcq))

(defn- split-column
  "Splits the CF:CQ expression into CF and CQ in bytes"
  [cfcq]
  (map ->bytes (str/split (name cfcq) #":" 2)))

(def ^:private split-column-memoize
  "Memoizing version of split-column.
  We use this when the argument is a keyword, since in that case it is almost
  certain that there aren't going to be too many of them."
  (memoize split-column))

(defmethod parse-column clojure.lang.Keyword
  [cfcq]
  (split-column-memoize cfcq))

(defmethod parse-column :default
  [cfcq]
  (split-column cfcq))

(defn ^TableName table-name-for
  "Returns TableName object for the name"
  [table-name]
  (if (instance? TableName table-name)
    table-name
    (TableName/valueOf (name table-name))))

