(ns utilities.forms.core
    (:require [hiccup.core :refer [html h]]
              [clojure.string :as s]
              [utilities.forms.field :as field]
              [utilities.forms.checks :as v]
              [utilities.exceptions :as ex]))


(defn Form
    "Gives a map with empty keywords:
    :clean? :error-msg :data"
    [fields checks]
    {:fields    fields
     :checks    checks
     :clean?    nil
     :error-msg nil
     :data      nil})


(defn FormField
    "Gives a map with empty keywords :clean? :error-msg"
    [name {:keys [field-type label value attrs help-msg optional? checks]
           :or   {field-type :input
                  label      (s/capitalize name)}}]
    {:field-type    field-type
     :name          name
     :label         label
     :value         value
     :attrs         attrs
     :help-msg      help-msg
     :checks        (or checks
                        (when-not optional?
                            (list v/not-blank?)))
     :clean?        nil
     :error-msg     nil})


(defn render-form
    "Renders all form fields with errors if any"
    [form]
    (let [fields-html (map field/render
                           (:fields form))
          msg-html (if-let [msg (:error-msg form)]
                       (html [:div.alert.error (h msg)])
                       "")]
        (str msg-html (s/join "\n" fields-html))))


(defn validate-form
    "First validates all the fields and then runs additional form checks.
    If everything passes, then adds :clean? true,
    else adds :error-msg to the form."
    [form data]
    (let [validated-fields (map (fn [{:keys [name] :as f}]
                                    (field/validate f (get data name)))
                                (:fields form))
          keyword-data (into {} (map (fn [{:keys [name value]}]
                                         {(keyword name) value})
                                     validated-fields))
          checks-result (if-not (every? :clean? validated-fields)
                            {:error-msg "Errors in input"}
                            (try
                                (when-let [checks (:checks form)]
                                    (every? (fn [check] (check keyword-data))
                                            checks)
                                    {:clean? true})
                                (catch Exception e
                                    (if (ex/cause? e :validation-error)
                                        {:error-msg (ex/message e)}
                                        (throw e)))))]
        (merge form
               {:fields validated-fields}
               {:data keyword-data}
               checks-result)))


(defn validate-on-post
    "Validates given Form type with the form data if request method is POST"
    [form request]
    (let [post? (= (:request-method request) :post)]
        (if post?
            (validate-form form (:form-params request))
            form)))
