(ns utilities.forms.field
    (:require [hiccup.core :refer [html h]]
              [utilities.exceptions :as ex]))


(defn- input-id
    "Creates input id for given field"
    [field]
    (or (:id (:attrs field))
        (str "id_" (:name field))))


(defn- field-attrs
    "Returns map of attributes for field html element"
    [default-attrs field]
    (merge default-attrs (:attrs field)))


(defn- label-html
    "returns label for given field"
    [field]
    (html [:label {:for (input-id field)}
           (h (:label field))
           (when (:optional? field) (html [:span.helptext "- Optional"]))
           (when-let [text (:error-msg field)] (html [:div.error text]))
           (when-let [text (:help-msg field)] (html [:br] [:span.helptext text]))]))


(defmulti render
          "Renders html for the given field based on its input type"
          :field-type)

(defmethod render :checkbox
    [{:keys [name value]
      :as field}]
    (let [id (input-id field)
          attrs (field-attrs {:id id, :name name, :class "u-full-width"}
                             field)]
        (html (label-html field)
              [:textarea attrs (h value)])))

(defmethod render :textarea
    [{:keys [name value] :as field}]
    (let [id (input-id field)
          attrs (field-attrs {:id id, :name name, :class "u-full-width"}
                             field)]
        (html (label-html field)
              [:textarea attrs (h value)])))

(defmethod render :default
    [{:keys [name value] :as field}]
    (let [id (input-id field)
          attrs (field-attrs {:id id, :type "text", :name name, :value value
                              :class "u-full-width"}
                             field)]
        (html (label-html field)
              [:input attrs])))


(defn- validator-result
    "Returns {:clean? true} if all the validators pass on value
    Else returns {:error-msg msg}"
    [checks value]
    (try (every? (fn [check] (check value))
                 checks)
         {:clean? true}
        (catch Exception e
            (if (ex/cause? e :validation-error)
                {:error-msg (ex/message e)}
                (throw e)))))


(defn validate [field value]
    (merge field
           {:value value}
           (validator-result (:checks field) value)))
