(ns utilities.forms.checks
  (:require [utilities.exceptions :as ex]
            [clojure.string :as s]))

; Sources: https://docs.djangoproject.com/en/2.1/_modules/django/core/validators/#validate_ipv46_address


(defn not-blank?
  "Returns true if the stripped value is a string of over 1 character"
  [value]
  (if (and (string? value)
           (not (empty? (s/trim value)))
           (not (nil? value)))
    true
    (ex/raise :validation-error "This is a required field")))


(defn email?
  "Returns true if given value is an email else raises error.

  Check is based on bouncer.validators which checks if there's
  only a single '@' and at least one point after the '@'
  "
  [value]
  (if (and (not-blank? value)
           (re-matches #"^[^@]+@[^@\\.]+[\\.].+" value))
    true
    (ex/raise :validation-error "Please provide a valid email")))


(defn min-length?
  "Returns true if the mininum length is satified"
  [length value]
  (when (>= (count value) length)
    true
    (ex/raise :validation-error (str "Value must be minimum "
                                     length " characters"))))
