(ns manager
  (:require [clojure.string :as s]
            [clojure.java.shell :as shell]
            [utilities.files :as f]
            [inflections.core :as i]
            [clojure.java.io :as io]
            [clj-jgit.porcelain :as g]))


(def resources (list "src/utilities" "doc" "test/utilities"))

(def template-path "resources/leiningen/new/jurassic_web")

(defn in?
  [item coll]
  (some #(= item %) coll))


(defn folder-name
  [path]
  (last (s/split path #"/")))

(defn exists
  "Returns true if the given file or directory exists"
  [path]
  (.exists (io/file path)))


(defn list-files
  "Returns names of files in the given directory or its sub-directories"
  [dir-path]
  (let [root  (io/file dir-path)
        root-path (.getPath root)]
    (seq (for [f (file-seq root)
               :when (.isFile f)]
           (s/replace (.getPath f) root-path "")))))


(defn delete-dir
  "Deletes given directory path"
  [dir]
  (let [root        (io/file dir)
        root-path   (.getPath root)
        contents    (file-seq root)
        sub-dirs    (remove #(or (.isFile %)
                                 (= (.getPath %) root-path))
                            contents)
        root-files  (filter #(and (.isFile %)
                                  (= (.getParent %) root-path))
                            contents)]
    (println "ROOT" root-path)
    (println "SUB" sub-dirs)
    (println "FILES" root-files)
    (doseq [d sub-dirs]
      (delete-dir (.getPath d)))
    (doseq [f root-files]
      (io/delete-file f))
    (io/delete-file dir)))


(defn copy-file
  "Copy file from src to destination"
  [src dest]
  (io/make-parents dest)
  (io/copy (io/file src) (io/file dest)))


(defn copy-dir
  "Copies given directory from source to destination"
  [src dest]
  (if (exists dest)
    (delete-dir dest))
  (doseq [f (list-files src)]
    (copy-file (f/path src f) (f/path dest f))))


(defn missing-in-source?
  "Returns true if not destination file is missing in source"
  [src-path dest-path]
  (let [sources       (set (list-files src-path))
        destinations  (set (list-files dest-path))
        diff          (clojure.set/difference destinations sources)]
    (when (seq diff)
      (throw (Exception.
               (str diff " files in destination but not in source"))))))


(defn hard-link-file
  "Creates hard link for src file to given destination"
  [src-file dest-file]
  (println (shell/sh "ln" "-vf" src-file dest-file)))


(defn hard-link-dir
  "Creates hard links for all files in given directory"
  [src-dir dest-dir]
  (when-not (missing-in-source? src-dir dest-dir)
    (seq (for [f (list-files src-dir)]
           (hard-link-file (f/path src-dir f)
                           (f/path dest-dir f))))))

(defn- sh
  [& args]
  (println "Calling " args)
  (let [c   (apply shell/sh args)]
    (println c)

    (if-not (empty? (:err c))
      (throw (Exception. ^String (str "Command Err: " (:err c)))))

    (if-not (= (:exit c) 0)
      (throw (Exception. ^String (str "Command Err - exited with " (:exit c)))))

    c))


(defn git-branch
  []
  (let [out         (:out (sh "git" "branch"))
        names-*     (map s/trim (s/split-lines out))
        current-*   (first (filter #(s/starts-with? % "* ") names-*))
        names       (remove #(= current-* %) names-*)
        current     (s/replace-first current-* "* " "")
        branches    (cons current names)]
    (println "BRANCHES" branches)
    branches))


(defn git-create-or-checkout
  "Create a git branch with given name"
  [repo project-name]
  (if-not (in? project-name (git-branch))
    (sh "git" "branch" project-name))
  (g/git-checkout repo project-name))


(defn pull-from-project
  "Creates a git branch to track project-path"
  [project-path project-name]
  (println "Project path" project-path)
  (let [repo          (g/load-repo ".")]
    (g/git-checkout repo "master")

    ; switch to project branch
    (git-create-or-checkout repo project-name)

    ; merge master
    (sh "git" "merge" "master")

    ; copy files from project
    (doseq [r resources]
      (copy-dir (f/path project-path r) (f/path template-path r)))

    ; do a commit
    (sh "git" "add" "." "-A")
    (sh "git" "commit" "-m" (str "Added updates from " project-path))

    ; reverse merge in master
    (g/git-checkout repo "master")
    (sh "git" "merge" project-name)
    (sh "git" "diff" "^HEAD~1" "--summary")
    (println "Copied updates from the project"
             "do a git diff"
             "replicate those changes in jurassic_web"
             "update version and release"
             "do push-to-project")))


(defn push-to-project
  "Pushes latest jurassic template to destination project"
  [project-path project-name]
  (let [repo  (g/load-repo ".")]
    (git-create-or-checkout repo project-name)
    (sh "git" "merge" "master")

    ; stop if differences from master as it needs to incorporated in this template
    (if-not (empty? (:out (sh "git" "diff" "master")))
      (throw (Exception. ^String (str "The project branch differs from master.\n"
                                      "INCORPORATE PROJECT BRANCH IN MASTER"))))

    (sh "lein" "new" "jurassic-web" project-name
        "--to-dir" project-path "--force")

    (g/git-checkout repo "master")))
