(ns utilities.db-test
  (:require [utilities.db :refer :all]
            [clojure.test :refer :all]
            [clojure.string :as s]))


(deftest test-builder
  (testing "various SQLs"
    (are [q r]
      (= (sql q) r)
      (query "books" {})  '("SELECT *\nFROM books")))

  (testing "no select"
    (let [q (query "company"
                   {:where "id = ::id"}
                   {:id nil})
          s (sql q)
          expected [(str "SELECT *\n"
                         "FROM company\n"
                         "WHERE id = ?") nil]]
      (is (= s expected))))

  (testing "query building"
    (let [q (query "users"
                   {:select "id, name"
                    :where "email like ::user-email"}
                   {:user-email "foo%"})
          s (sql q)
          expected [(str "SELECT id, name\n"
                         "FROM users\n"
                         "WHERE email like ?")
                    "foo%"]]
      (is (= s expected))))

  (testing "chained query building"
    (let [q (query "users"
                   {:select "id, name"
                    :where "email like ::user-email"}
                   {:user-email "foo%"})
          q (query q
                   {:select "about AS ::col-name"
                    :where "about NOT LIKE ::private"}
                   {:col-name "user-about"
                    :private "%private%"})
          s (sql q)
          expected [(str "SELECT id, name, about AS ?\n"
                         "FROM users\n"
                         "WHERE email like ? AND about NOT LIKE ?")
                    "user-about" "foo%" "%private%"]]
      (is (= s expected)))))


(deftest unit-tests
  (testing "query-params"
    (let [q (query  "foobar"
                    {:select "id, name, foo AS ::col"
                     :where ["some_col > ::val" "another_col > ::val2"]
                     :group-by "bar"
                     :having "::col > ::group_val"}
                    {:val 1, :val2 2, :col "baz", :group_val 5})
          expected  ["some_col > ? AND another_col > ?" 1 2]
          res       (query-params (s/join " AND " (:where q)) (:params q))]
      (is (= res expected)))

    (is (= (query-params "id <> ::id" {})
           ["id <> ?" ""]))))