(ns utilities.forms.core
  (:require [hiccup.core :refer [html h]]
            [clojure.string :as s]
            [utilities.forms.components :as field]
            [utilities.forms.checks :as v]
            [utilities.exceptions :as ex]))

; Field types

(def input-field ::field/input)
(def custom-field ::field/custom-field)
(def textarea-field ::field/textarea)
(def checkbox-field ::field/checkbox)
(def hidden-field ::field/hidden)
(def select-field ::field/select)



(defn form-field
  "Generates a field map with proper defaults"
  ([name] (form-field name {}))

  ([name {:keys [field-type label attrs help-msg optional?
                 renderer options]}]

   (when (and (= field-type custom-field)
              (not renderer))
     (throw (Exception. (str "Renderer not provided for custom field " name))))

   (when (and renderer (not= field-type custom-field))
     (throw (Exception. "Renderer is supported only for custom field")))

   (when (and (= field-type select-field)
              (not (map? options)))
     (throw (Exception. (str "No options provided for select field " name))))

   {:field-type    (or field-type input-field)
    :name          (keyword name)
    :label         (or label (s/capitalize name))
    :attrs         attrs
    :help-msg      help-msg
    :optional?     optional?
    :renderer      renderer
    :options       options}))


(defn render-form
  "renders form with errors and data"
  [fields data errors]
  (let [fields-html   (for [field fields]
                        (let [key (:name field)]
                          (field/render field (key data) (key errors))))
        form-msg      (when-let [msg (::form errors)]
                        (html [:div.alert.error (h msg)]))]
    (str form-msg (s/join "\n" fields-html))))


(defn get-fields-data
  "Returns map of field names and corresponding data from data map"
  [fields data]
  (into {} (for [field fields]
             (let [key (:name field)]
               [key (get data key
                         (get data (name key)))]))))


(defn get-post-data
  "Parses post data from request and returns matching map based on fields
  If initial data is provided, then post data is merged into it."
  ([fields request]
   (let [post-data  (:form-params request)]
     (if-not (empty? post-data)
       (get-fields-data fields post-data)))))


(defn- to-coll
  "Converts a single function into a list of collections"
  [x]
  (if (and x (not (coll? x)))
    (list x)
    x))


(defn- get-checks
  "Returns checks for given field including non-blank"
  [field all-field-checks]
  (let [field-checks  (get all-field-checks (:name field))
        field-checks  (to-coll field-checks)]
    (if (:optional? field)
      field-checks
      (cons v/not-blank? field-checks))))


(defn- get-err
  "Returns first error message if any check fails on value"
  [value checks]
  (try
    (every? #(% value) checks)
    nil
    (catch Exception e
      (if (ex/cause? e :validation-error)
        (ex/message e)
        (throw e)))))


(defn- get-field-errors
  "Returns map of field name and error for each fail"
  [fields data field-checks]
  (let [field-checks  (into {}
                            (for [field fields]
                              [(:name field) (get-checks field field-checks)]))]
    (into {} (for [[key checks] field-checks]
               (when-let [err (get-err (key data) checks)]
                 [key err])))))


(defn- get-form-errors
  "Returns map of form error with key ::form"
  [data form-checks]
  (let [error  (get-err data (to-coll form-checks))]
    (when error
      {::form error})))


(defn get-errors
  "returns map of field errors and ::form error by running the given checks
  returns :no-errors if all tests pass
  returns :no-data in case of empty data"
  [fields data field-checks form-checks]
  (if (empty? data)
    :no-data
    (let [field-errors  (get-field-errors fields data field-checks)
          errors        (if (empty? field-errors)
                          (get-form-errors data form-checks)
                          field-errors)]
      (if (empty? errors)
        :no-errors
        errors))))
