(ns utilities.paging
  (:require [utilities.db :as db]
            [utilities.routing :as routes]
            [hiccup.core :refer [html h]]))


(defn- get-number
  ([n] (get-number n 1))
  ([n default]
   (try
     (Integer. n)
     (catch Exception e
       default))))


(defn page
  "Limits and offsets results based on given page"
  [q page-size page-number]
  (let [page-number       (get-number page-number)
        results-count     (db/count! q)
        pages-count       (Math/ceil (/ results-count page-size))
        page-number       (cond
                            (> page-number pages-count) 1
                            (< page-number 1) 1
                            :else page-number)
        offset            (let [n (* (- page-number 1) page-size)]
                            (if (< n 0)
                              0
                              n))
        results           (db/all! (db/query q {:limit page-size
                                                :offset offset}))]
    {:results       results
     :results-count results-count
     :pages-count   pages-count
     :start         offset
     :page-number   page-number}))


(defn request-page
  "Limits and offsets results based on given request"
  [request q page-size max-page-size]
  (throw (Exception. "to implement")))


(defn- page-link
  ([request n] (page-link request n n))

  ([request n name]
   (when n
     (html [:a {:href (routes/get-params request {:page n})} (h name)]))))


(defn page-numbers
  "Renders page numbers for given page results"
  [request page]
  (let [pages-count   (:pages-count page)
        page-number   (:page-number page)
        prev-page     (if (> page-number 1)
                        (dec page-number))
        start-pages   (if (and prev-page (> prev-page 1))
                        1)
        next-page     (if (> pages-count page-number)
                        (inc page-number))
        end-pages     (if (and next-page (> pages-count next-page))
                        pages-count)]
    (html (page-link request start-pages)
          (page-link request prev-page)
          page-number
          (page-link request next-page)
          (page-link request end-pages))))
