(ns utilities.routing
  (:require [utilities.templates :as t]
            [compojure.core :as c]
            [utilities.exceptions :as ex]
            [clojure.string :as s]
            [ring.util.codec :as codec]))


; Define urls as nil if not already defined
; names-p is map of url name and its broken path
(def ^:dynamic *names-p* nil)


(defn names-broken-path
  "Returns a map of url name and broken path for given urls"
  [urls]
  (into {} (for [[name path _] (partition 3 urls)]
             [name (t/break-text path)])))


(defn set-urls
  "Sets urls to given urls for reverse handling"
  [urls]
  (def ^:dynamic *names-p* (names-broken-path urls)))


(defn get-url
  "Generates a function which provides url for given name and context"
  ([name] (get-url name {}))

  ([name context]
   (let [path          (get *names-p* name)
         keys          (filter keyword? path)
         missing-keys  (seq (remove #(get context %) keys))]
     (if-not path (ex/raise ::routing-error
                            (str "URL with name " name " not found")))

     (if missing-keys
       (ex/raise ::routing-error
                 (str (s/join ", " missing-keys)
                      " not provided for " name " url")))

     (t/fill-broken-text path context))))


(defn- get-vals
  "Returns collection of values from map for given keys"
  [coll keys]
  (for [key keys]
    (get coll key)))


(defn handlers
  "Returns handlers to be passed to compojure.routes for given urls vector"
  [urls]
  (let [names-p (names-broken-path urls)]
    (for [[n _ f] (partition 3 urls)]
      (let [p     (get names-p n)
            args  (filter keyword? p)
            ; replace our placeholder with compujure-keyword syntax
            path  (t/fill-broken-text p (zipmap args
                                                (map str args)))]
        (c/ANY path request
          (if args
            (apply f request
                   (get-vals (:route-params request) args))
            (f request)))))))


(defn get-params
  "Returns get link after updating current query params with new data"
  [request data]
  (let [params   (:query-params request)
        params   (merge params data)]
    (str "?" (codec/form-encode params))))
