(ns utilities.forms.core
  (:require [hiccup.core :refer [html h]]
            [clojure.string :as s]
            [utilities.forms.components :as field]
            [utilities.forms.checks :as v]
            [utilities.exceptions :as ex]))

; Field types

(def input-field ::field/input)
(def custom-field ::field/custom-field)
(def textarea-field ::field/textarea)
(def checkbox-field ::field/checkbox)
(def hidden-field ::field/hidden)
(def select-field ::field/select)


(defn form-field
  "Generates a field map with proper defaults"
  ([field-name] (form-field field-name {}))

  ([field-name {:keys [field-type label attrs help-msg optional?
                       renderer options]}]

   (when (and (= field-type custom-field)
              (not renderer))
     (throw (Exception.
              (str "Renderer not provided for custom field " field-name))))

   (when (and renderer (not= field-type custom-field))
     (throw (Exception. "Renderer is supported only for custom field")))

   (when (and (= field-type select-field)
              (not (map? options)))
     (throw (Exception.
              (str "No options provided for select field " field-name))))

   {:field-type    (or field-type input-field)
    :name          (keyword field-name)
    :label         (or label (s/capitalize (name field-name)))
    :attrs         attrs
    :help-msg      help-msg
    :optional?     optional?
    :renderer      renderer
    :options       options}))


(defn render-form
  "renders form with errors and data"
  [fields data errors]
  (let [fields-html   (for [field fields]
                        (let [key (:name field)]
                          (field/render field (key data) (key errors))))
        form-msg      (when-let [msg (::form errors)]
                        (html [:div.alert.error (h msg)]))]
    (str form-msg (s/join "\n" fields-html))))


(defn get-fields-data
  "Returns map of field names and corresponding data from data map"
  [fields data]
  (into {} (for [field fields]
             (let [key (:name field)]
               [key (get data key
                         (get data (name key)))]))))


(defn get-post-data
  "Parses post data from request and returns matching map based on fields
  If initial data is provided, then post data is merged into it."
  ([fields request]
   (let [post-data  (:form-params request)]
     (if-not (empty? post-data)
       (get-fields-data fields post-data)))))


(defmacro get-err
  "Returns validation error message from given check"
  [check]
  `(try
     ~check
     nil
     (catch Exception e#
       (if (ex/cause? e# :validation-error)
         (ex/message e#)
         (throw e#)))))


(defn check-non-blank
  "Returns map of errors for blank non-optional fields"
  [fields data]
  (let [fields  (remove :optional? fields)]
    (reduce (fn [errors field]
              (let [name (:name field)]
                (if-let [err (get-err (v/not-blank? (get data name)))]
                  (assoc errors name err)
                  errors)))
            {}
            fields)))


(defmacro get-field-errors
  "Returns map of all field name and field error"
  [initial-errors field-checks]
  (let [errors  (for [[name check] field-checks]
                   `(when-let [err#  (or ~(get initial-errors name)
                                         (get-err ~check))]
                      {~name err#}))]
    `(merge ~initial-errors ~@errors)))


(defmacro get-form-errors
  "Returns first error in form-checks"
  [form-checks]
  (let [form-checks  (for [check form-checks]
                       `(get-err ~check))]
    `(when-let [err# (or ~@form-checks)]
       {::form err#})))


(defmacro checks
  "Runs the given checks in given order.
  If any field's check fails, then its consecutive checks are skipped.
  ::form checks are run only if no field checks are there."
  [initial-errors & clauses]
  (let [[field-checks form-checks]  (split-with #(not= % ::form) clauses)
        field-checks (when-not (empty? field-checks)
                       (apply hash-map field-checks))
        form-checks  (when-not (empty? form-checks)
                       (apply hash-map form-checks))]
    `(let [field-errors#  (get-field-errors ~initial-errors ~field-checks)
           errors#        (if (empty? field-errors#)
                            (get-form-errors ~form-checks)
                            field-errors#)]
       (if (empty? errors#)
         :no-errors
         errors#))))
